<?php
/**
 * Handler for listing directory contents.
 * Returns a JSON array with file names, sizes, and subdirectory names.
 *
 * @param int   $userid   User ID (currently unused, kept for signature compatibility)
 * @param array $req      Request parameters (currently unused)
 * @param string $requestPath Absolute path to the directory to list
 * @param array &$resp   Response array to be populated
 * @return bool  True on success, false on error (error details placed in $resp)
 */
function handlerequest_fileserver_listdir($userid, $req, $requestPath, &$resp)
{
    if (isDebug()) error_log("fileserver.inc.listdir.php: directory list request: $userid, $requestPath");
    if (!fileServerListDir($userid, $req, $requestPath, $itemsjson, $error)) {
        ozerror(__FILE__, $error, $resp, $error);
        return false;
    }

    $resp["status"] = "OK";
    $resp["errormessage"] = "";
    $resp["items"] = $itemsjson;
    return true;
}

/**
 * Internal helper that builds the JSON payload.
 *
 * @param int   $userid   User ID (unused)
 * @param array $req      Request parameters (unused)
 * @param string $requestPath Directory to list
 * @param string &$itemsjson JSON-encoded result
 * @param string &$error    Error message on failure
 * @return bool  True on success, false on failure
 */
function fileServerListDir($userid, $req, $requestPath, &$itemsjson, &$error)
{
    // Ensure the path exists and is a directory
    if (!is_dir($requestPath)) {
        $itemsjson = json_encode([]);
        return true;
    }

    // Use the existing directory listing helper (ozlsdir or its fallback)
    $listOk = ozlsdir($requestPath, $files, $dirs, $error);

    if (!$listOk) {
        $error = "Cannot list items. $error";
        return false;
    }

    $result = [];

    // Add files with their sizes
    foreach ($files as $file) {
        $fullPath = $requestPath . '/' . $file;
        $size = is_file($fullPath) ? filesize($fullPath) : 0;
        $created = date('c', filectime($fullPath));
        $modified = date('c', filemtime($fullPath));
        $result[] = [
            "name" => $file,
            "size" => $size,
            "type" => "file",
            "created" => $created,
            "modified" => $modified
        ];
    }

    // Add subdirectories (size set to 0)
    foreach ($dirs as $dir) {
        $fullPath = $requestPath . '/' . $dir;
        $created = date('c', filectime($fullPath));
        $modified = date('c', filemtime($fullPath));
        $result[] = [
            "name" => $dir,
            "size" => 0,
            "type" => "directory",
            "created" => $created,
            "modified" => $modified
        ];
    }

    $itemsjson = json_encode($result);
    return true;
}
?>