<?php
// Set headers for JSON response
//header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle OPTIONS request for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}


// If GET request or no POST, show the HTML form
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OpenAI API Test</title>
    <style>
     
        
        .form-section, .response-section {
            padding: 30px;
        }
        
        .form-section {
            border-right: 1px solid #e0e0e0;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #4CAF50;
        }
        
        .form-group textarea {
            resize: vertical;
            min-height: 120px;
            font-family: 'Courier New', monospace;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .checkbox-group input[type="checkbox"] {
            width: auto;
            transform: scale(1.2);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        
        .btn {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
            color: white;
            border: none;
            padding: 15px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
            width: 100%;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(76, 175, 80, 0.3);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .btn:disabled {
            background: #ccc;
            cursor: not-allowed;
            transform: none;
        }
        
        .response-section {
            background: #f8f9fa;
        }
        
        .response-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .response-title {
            font-size: 1.5em;
            font-weight: 600;
            color: #333;
        }
        
        .status {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status.idle {
            background: #e0e0e0;
            color: #666;
        }
        
        .status.loading {
            background: #fff3cd;
            color: #856404;
        }
        
        .status.success {
            background: #d4edda;
            color: #155724;
        }
        
        .status.error {
            background: #f8d7da;
            color: #721c24;
        }
        
        .response-content {
            background: white;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            padding: 20px;
            min-height: 400px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            white-space: pre-wrap;
            overflow-x: auto;
        }
        
        .clear-btn {
            background: #6c757d;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .clear-btn:hover {
            background: #5a6268;
        }
        
        .help-text {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }

    </style>
</head>
<body>
    <div class="container">
       
        <div class="content">
            <div class="form-section">
                <form id="apiForm">
                    
                    <div class="form-group">
                        <label for="endpoint">API Endpoint *</label>
                        <select id="endpoint" name="endpoint" required>
                            <option value="">Select an endpoint</option>
                            <option value="http://llm.ozeki.hu:8001/v1" selected>http://llm.ozeki.hu:8001/v1</option>
                            <option value="http://video.ozeki.hu:8123/v1" selected>http://video.ozeki.hu:8123/v1</option>
                            <option value="http://localhost/v1" selected>http://localhost/v1</option>
                            <option value="https://api.openai.com/v1">https://api.openai.com/v1</option>
                            <option value="https://api.groq.com/openai/v1">https://api.groq.com/openai/v1</option>
                            <option value="https://api.anthropic.com/v1">https://api.anthropic.com/v1</option>
                            <option value="custom">Custom endpoint...</option>
                        </select>
                        <input type="url" id="customEndpoint" name="custom_endpoint" placeholder="Enter custom endpoint URL" style="display: none; margin-top: 10px;">
                        <div class="help-text">OpenAI API base URL (without /chat/completions)</div>
                    </div>

                    <div class="form-group">
                        <label for="apiKey">API Key *</label>
                        <input type="password" id="apiKey" name="api_key" value="sk-" required placeholder="sk-...">
                    </div>

                    
                    <div class="form-group">
                        <label for="model">Model *</label>
                        <input type="text" id="model" name="model" value="GLM-4.6" required placeholder="GLM-4.6">
                    </div>
                    
                    <div class="form-group">
                        <label for="message">Message *</label>
                        <textarea id="message" name="message" required placeholder='Enter your message here or JSON array of messages&#10;Example: [{"role": "user", "content": "Hello!"}]'>Where is Budapest?</textarea>
                        <div class="help-text">You can enter a simple message or JSON array of message objects</div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="temperature">Temperature</label>
                            <input type="number" id="temperature" name="temperature" value="1.0" min="0" max="2" step="0.1">
                            <div class="help-text">0.0 (deterministic) to 2.0 (creative)</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="maxTokens">Max Tokens</label>
                            <input type="number" id="maxTokens" name="max_tokens" placeholder="Optional" min="1" value="1500">
                            <div class="help-text">Leave empty for default</div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <div class="checkbox-group">
                            <input type="checkbox" id="stream" name="stream">
                            <label for="stream">Enable Streaming</label>
                        </div>
                        <div class="help-text">Stream the response as it's generated</div>
                    </div>
                    
                    <button type="submit" class="btn" id="submitBtn">Send Request</button>
                </form>
            </div>
            
            <div class="response-section">
                <div class="response-header">
                    <div class="response-title">Response</div>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <div class="status idle" id="status">Idle</div>
                        <button class="clear-btn" id="clearBtn">Clear</button>
                    </div>
                </div>
                <div class="response-content" id="responseContent">Response will appear here...</div>
            </div>
        </div>
    </div>

    <script>
        
        const form = document.getElementById('apiForm');
        const submitBtn = document.getElementById('submitBtn');
        const responseContent = document.getElementById('responseContent');
        const status = document.getElementById('status');
        const clearBtn = document.getElementById('clearBtn');
        const endpointSelect = document.getElementById('endpoint');
        const customEndpointInput = document.getElementById('customEndpoint');
        
        // Handle endpoint selection change
        endpointSelect.addEventListener('change', function() {
            if (this.value === 'custom') {
                customEndpointInput.style.display = 'block';
                customEndpointInput.required = true;
            } else {
                customEndpointInput.style.display = 'none';
                customEndpointInput.required = false;
            }
            
            // Update model based on selected endpoint
            const modelInput = document.getElementById('model');
            switch(this.value) {
                case 'http://llm.ozeki.hu:8001/v1':
                    modelInput.value = 'GLM-4.6';
                    break;
                case 'http://localhost/v1':
                    modelInput.value = 'GLM-4.6';
                    break;
                case 'http://video.ozeki.hu:8123/v1':
                    modelInput.value = 'unsloth/Qwen3-Coder-30B-A3B-Instruct-1M-GGUF/BF16/Qwen3-Coder-30B-A3B-Instruct-1M-BF16-00001-of-00002.gguf';
                    break;
                case 'https://api.openai.com/v1':
                    modelInput.value = 'gpt-3.5-turbo';
                    break;
                case 'https://api.groq.com/openai/v1':
                    modelInput.value = 'llama3-8b-8192';
                    break;
                case 'https://api.anthropic.com/v1':
                    modelInput.value = 'claude-3-haiku-20240307';
                    break;
                default:
                    // For Ozeki and custom endpoints, keep the default model or empty
                    if (this.value === '') {
                        modelInput.value = '';
                    }
            }
        });
        
        clearBtn.addEventListener('click', () => {
            responseContent.textContent = 'Response will appear here...';
            status.className = 'status idle';
            status.textContent = 'Idle';
        });
        
        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            console.log("Form submitted");

            const formData = new FormData(form);
            const data = Object.fromEntries(formData.entries());
            
            // Convert checkbox to boolean
            data.stream = formData.has('stream');
            
            // Handle custom endpoint
            if (data.endpoint === 'custom') {
                data.endpoint = data.custom_endpoint;
            }
            
            // Validate required fields
            if (!data.api_key || !data.endpoint || !data.model || !data.message) {
                responseContent.textContent = 'Error: Please fill in all required fields';
                status.className = 'status error';
                status.textContent = 'Error';
                return;
            }
            
            // Update UI
            submitBtn.disabled = true;
            submitBtn.textContent = 'Sending...';
            status.className = 'status loading';
            status.textContent = 'Loading';
            responseContent.textContent = '';
            rawResponse = '';
            
            try {
                // Prepare OpenAI API request payload
                const payload = {
                    model: data.model,
                    messages: [],
                    temperature: parseFloat(data.temperature) || 1.0,
                    stream: data.stream
                };
                
                // Add max_tokens if provided
                if (data.max_tokens) {
                    payload.max_tokens = parseInt(data.max_tokens);
                }
                
                // Parse messages - handle both simple text and JSON array
                try {
                    // Try to parse as JSON array first
                    const parsedMessages = JSON.parse(data.message);
                    if (Array.isArray(parsedMessages)) {
                        payload.messages = parsedMessages;
                    } else {
                        throw new Error('Not an array');
                    }
                } catch {
                    // If not valid JSON array, treat as simple user message
                    payload.messages = [
                        {
                            role: "user",
                            content: data.message
                        }
                    ];
                }
                
                // Make OpenAI API request
                const apiEndpoint = data.endpoint.endsWith('/') ? 
                    data.endpoint + 'chat/completions' : 
                    data.endpoint + '/chat/completions';
                
                const response = await fetch(apiEndpoint, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${data.api_key}`,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify(payload)
                });
                
                if (data.stream) {
                    // Handle streaming response
                    status.className = 'status success';
                    status.textContent = 'Streaming';
                    
                    const reader = response.body.getReader();
                    const decoder = new TextDecoder();
                    
                    while (true) {
                        const { done, value } = await reader.read();
                        if (done) break;
                        
                        const chunk = decoder.decode(value, { stream: true });
                        const lines = chunk.split('\n');
                        
                        for (const line of lines) {
                            if (line.startsWith('data: ')) {
                                const data = line.slice(6);
                                if (data === '[DONE]') {
                                    break;
                                }
                                // Fetch as plain text first, then parse as JSON
                                try {
                                    // In this case, we're already working with the parsed data from the streaming response
                                    // but to demonstrate the principle, we would separate fetch and parse steps
                                    const parsed = JSON.parse(data);
                                    const content = parsed.choices?.[0]?.delta?.content || '';
                                    if (content) {
                                        responseContent.textContent += content;
                                    }
                                } catch (e) {
                                    // Display raw data when JSON parsing fails
                                    responseContent.textContent += '\n[RAW DATA - JSON Parse Error]: ' + data;
                                }
                            }
                        }
                    }
                } else {
                    // Handle normal response
                    try {
                        // Fetch as plain text first, then parse as JSON
                        rawResponse = await response.text();
                        
                        // Check if raw response is empty
                        if (!rawResponse) {
                            status.className = 'status error';
                            status.textContent = 'Error';
                            responseContent.textContent = 'Received empty response from the API';
                            return;
                        }
                        
                        const result = JSON.parse(rawResponse);
                        
                        if (!response.ok) {
                            status.className = 'status error';
                            status.textContent = 'Error';
                            responseContent.textContent = JSON.stringify(result, null, 2);
                        } else {
                            status.className = 'status success';
                            status.textContent = 'Success';
                            
                            // Extract and display the assistant's response
                            if (result.choices && result.choices.length > 0) {
                                const assistantMessage = result.choices[0].message?.content || 'No content received';
                                responseContent.textContent = assistantMessage;
                            } else {
                                responseContent.textContent = JSON.stringify(result, null, 2);
                            }
                        }
                    } catch (jsonError) {
                        // If JSON parsing fails, display the raw response text
                        status.className = 'status error';
                        status.textContent = 'JSON Parse Error';
                        responseContent.textContent = '[RAW RESPONSE - JSON Parse Error]:\n' + rawResponse;
                    }
                }
            } catch (error) {
                status.className = 'status error';
                status.textContent = 'Error';
                responseContent.textContent = 'Network error: ' + error.message;
                console.error('Request failed:', error);
            } finally {
                submitBtn.disabled = false;
                submitBtn.textContent = 'Send Request';
            }
        });
    </script>
</body>
</html>
