<?php
/**
 * Handle request to modify php.ini for enabling cURL.
 *
 * @param string $userid
 * @param array  $req   Request data
 * @param array  &$resp Response data (by reference)
 * @return bool
 */
function handlerequest_phpconfig($userid, $req, &$resp)
{
    error_log("servercontrol_phpconfig.php: handlerequest_phpconfig called for user $userid with request " . json_encode($req));

    // Resolve php.ini path
    $iniPath = locatePhpIni();
    $phpStatus = 'not_found';

    if ($iniPath !== false) {
        $phpStatus = phpConfigEnableCurl($userid, $req, $resp, $error, $iniPath, $phpStatus) ? $phpStatus : 'error';
    }

    // Build response
    $error = $error ?? '';
    $resp["status"]        = "OK";
    $resp["errormessage"]  = $error;
    $resp["php.ini"]       = $iniPath !== false ? $iniPath : null;
    $resp["php_status"]    = $phpStatus;

    return true;
}

/**
 * Locate the php.ini file using php_ini_loaded_file().
 *
 * @return string|false Full path to php.ini if found, otherwise false.
 */
function locatePhpIni()
{
    $path = php_ini_loaded_file();
    return $path !== false ? $path : false;
}

/**
 * Enable the cURL extension in php.ini.
 *
 * @param string $userid
 * @param array  $req
 * @param array  $resp
 * @param string &$error   Error message (by reference)
 * @param string $iniPath  Path to php.ini (optional)
 * @param string &$status  Status string (by reference)
 * @return bool
 */
function phpConfigEnableCurl($userid, $req, $resp, &$error, $iniPath = null, &$status = null)
{
    if ($iniPath === null) {
        $iniPath = locatePhpIni();
    }
    if ($iniPath === false) {
        $error  = 'Unable to resolve php.ini path.';
        $status = 'not_found';
        return false;
    }

    $lines = @file($iniPath);
    if ($lines === false) {
        $error  = 'Failed to read php.ini.';
        $status = 'error';
        return false;
    }

    $changed = false;
    foreach ($lines as $i => $line) {
        // Match lines like ";extension=curl" or ";extension=php_curl.dll"
        if (preg_match('/^\s*;?\s*extension\s*=\s*(php_curl\.dll|curl)\s*$/i', $line)) {
            // Uncomment if commented
            $newLine = preg_replace('/^\s*;/', '', $line);
            if ($newLine !== $line) {
                $lines[$i] = $newLine;
                $changed   = true;
            }
        }
    }

    if ($changed) {
        // Create backup with timestamp suffix
        $backupPath = $iniPath . '_' . date('Ymd_His');
        if (!@copy($iniPath, $backupPath)) {
            $error  = 'Failed to create backup of php.ini.';
            $status = 'error';
            return false;
        }

        $result = @file_put_contents($iniPath, implode('', $lines));
        if ($result === false) {
            $error  = 'Failed to write updated php.ini.';
            $status = 'error';
            return false;
        }

        $status = 'updated';
        error_log("servercontrol_phpconfig.php: php.ini updated at $iniPath (backup created at $backupPath)");
    } else {
        $status = 'unchanged';
        error_log("servercontrol_phpconfig.php: php.ini unchanged at $iniPath");
    }

    return true;
}
?>