<?php


function handlerequest_apacheconfig($userid,$req,&$resp)
{
    error_log("servercontrol_apacheconfig.php: handlerequest_apacheconfig called for user $userid with request ".json_encode($req));
    // Resolve configuration file paths
    $httpdPath   = locateHttpdConf();
    $vhostsPath  = locateVhostsConf();

    // Update httpd.conf (mod_rewrite, proxy, etc.)
    $httpdStatus = 'not_found';
    if ($httpdPath !== false) {
        $httpdStatus = apacheConfigEnableModRewrite($userid,$req,$resp,$error,$httpdPath,$httpdStatus) ? $httpdStatus : 'error';
    error_log("servercontrol_apacheconfig.php: httpd.conf update status: $httpdStatus");
    }

    // Update httpd-vhosts.conf (rewrite rules)
    $vhostsStatus = 'not_found';
    if ($vhostsPath !== false) {
        $vhostsStatus = apacheVhostsAddRewrite($userid,$req,$resp,$error,$vhostsPath,$vhostsStatus) ? $vhostsStatus : 'error';
    error_log("servercontrol_apacheconfig.php: vhosts.conf update status: $vhostsStatus");
    }

    // Build response
    $error = "Apache config update success. $error";
    $resp["status"]        = "OK";
    $resp["errormessage"]  = $error;
    $resp["httpd.conf"]    = $httpdPath !== false ? $httpdPath : null;
    $resp["vhosts.conf"]   = $vhostsPath !== false ? $vhostsPath : null;
    $resp["httpd_status"]  = $httpdStatus;
    $resp["vhosts_status"] = $vhostsStatus;

    return true;
}

/**
 * Locate the httpd.conf file for a WAMP installation.
 *
 * @param string $baseDir Base WAMP directory (default "C:\wamp64").
 * @return string|false Full path to httpd.conf if found, otherwise false.
 */
function locateHttpdConf($baseDir = 'C:\\wamp64')
{
    $apacheBase = $baseDir . DIRECTORY_SEPARATOR . 'bin' . DIRECTORY_SEPARATOR . 'apache';
    // Find directories that match the Apache version pattern (e.g., apache2.4.51)
    $dirs = glob($apacheBase . DIRECTORY_SEPARATOR . 'apache*', GLOB_ONLYDIR);
    $bestConf = false;
    $bestVersion = null;
    foreach ($dirs as $dir) {
        $conf = $dir . DIRECTORY_SEPARATOR . 'conf' . DIRECTORY_SEPARATOR . 'httpd.conf';
        if (!file_exists($conf)) {
            continue;
        }
        // Extract version number from directory name, e.g., apache2.4.51 -> 2.4.51
        if (preg_match('/apache([\d\.]+)/', $dir, $matches)) {
            $version = $matches[1];
            if ($bestVersion === null || version_compare($version, $bestVersion, '>')) {
                $bestVersion = $version;
                $bestConf = $conf;
            }
        }
    }
    return $bestConf !== false ? $bestConf : false;
}

function apacheConfigEnableModRewrite($userid,$req,$resp,&$error,$confPath=null,&$status=null)
{
    // Determine path to httpd.conf
    if ($confPath === null) {
        $confPath = locateHttpdConf();
    }
    if ($confPath === false) {
        $error  = 'Unable to resolve httpd.conf path.';
        $status = 'not_found';
        return false;
    }

    // Read file
    $lines = @file($confPath);
    if ($lines === false) {
        $error  = 'Failed to read httpd.conf.';
        $status = 'error';
        return false;
    }

    $modules = ['mod_rewrite.so','mod_proxy.so','mod_proxy_http.so'];
    $changed = false;
    foreach ($lines as $i => $line) {
        foreach ($modules as $mod) {
            $pattern = '/^#\s*LoadModule\s+\w+_module\s+modules\/' . preg_quote($mod, '/') . '\s*$/';
            if (preg_match($pattern, $line)) {
                $lines[$i] = preg_replace('/^#\s*/', '', $line);
                $changed = true;
            }
        }
    }

    if ($changed) {
        // Create backup of original httpd.conf with timestamp
        $backupPath = $confPath . '_' . date('Ymd_His');
        if (!@copy($confPath, $backupPath)) {
            $error = 'Failed to create backup of httpd.conf.';
            $status = 'error';
            return false;
        }
        $result = @file_put_contents($confPath, implode('', $lines));
        if ($result === false) {
            $error  = 'Failed to write updated httpd.conf.';
            $status = 'error';
            return false;
        }
        $status = 'updated';
        error_log("servercontrol_apacheconfig.php: httpd.conf updated at $confPath (backup created at $backupPath)");
    } else {
        $status = 'unchanged';
        error_log("servercontrol_apacheconfig.php: httpd.conf unchanged at $confPath");
    }

    return true;
}

/**
 * Locate the httpd-vhosts.conf file for a WAMP installation.
 *
 * @param string $baseDir Base WAMP directory (default "C:\wamp64").
 * @return string|false Full path to httpd-vhosts.conf if found, otherwise false.
 */
function locateVhostsConf($baseDir = 'C:\\wamp64')
{
    $apacheBase = $baseDir . DIRECTORY_SEPARATOR . 'bin' . DIRECTORY_SEPARATOR . 'apache';
    $dirs = glob($apacheBase . DIRECTORY_SEPARATOR . 'apache*', GLOB_ONLYDIR);
    $bestConf = false;
    $bestVersion = null;
    foreach ($dirs as $dir) {
        $conf = $dir . DIRECTORY_SEPARATOR . 'conf' . DIRECTORY_SEPARATOR . 'extra' . DIRECTORY_SEPARATOR . 'httpd-vhosts.conf';
        if (!file_exists($conf)) {
            continue;
        }
        if (preg_match('/apache([\d\.]+)/', $dir, $matches)) {
            $version = $matches[1];
            if ($bestVersion === null || version_compare($version, $bestVersion, '>')) {
                $bestVersion = $version;
                $bestConf = $conf;
            }
        }
    }
    return $bestConf !== false ? $bestConf : false;
}

/**
 * Add required rewrite rules to VirtualHost sections in httpd-vhosts.conf.
 *
 * @param string $userid
 * @param array $req
 * @param array $resp
 * @param string &$error
 * @param string|null $vhostsPath
 * @return bool
 */
function apacheVhostsAddRewrite($userid, $req, $resp, &$error, $vhostsPath = null, &$status = null)
{
    if ($vhostsPath === null) {
        $vhostsPath = locateVhostsConf();
    }
    if ($vhostsPath === false) {
        $error  = 'Unable to resolve httpd-vhosts.conf path.';
        $status = 'not_found';
        return false;
    }

    $lines = @file($vhostsPath);
    if ($lines === false) {
        $error  = 'Failed to read httpd-vhosts.conf.';
        $status = 'error';
        return false;
    }

    $rewriteLines = [
        "RewriteEngine On\n",
        "#LogLevel alert rewrite:trace6\n",
        "RewriteRule ^/v1/(.*) /ozeki/index.php?srv=aigate&api=gateway&path=/v1/$1\n"
    ];

    $changed = false;
    $insideVhost = false;
    $vhostStartLine = null;

    foreach ($lines as $i => $line) {
        if (preg_match('/<VirtualHost\s+[^>]*:(80|443)>/', $line)) {
            $insideVhost = true;
            $vhostStartLine = $i;
            continue;
        }
        if ($insideVhost && preg_match('/<\/VirtualHost>/', $line)) {
            // Extract current section text
            $section = array_slice($lines, $vhostStartLine, $i - $vhostStartLine + 1);
            $sectionText = implode('', $section);
            $missing = [];
            foreach ($rewriteLines as $rw) {
                if (strpos($sectionText, trim($rw)) === false) {
                    $missing[] = $rw;
                }
            }
            if (!empty($missing)) {
                // Insert missing lines just before the closing tag
                array_splice($lines, $i, 0, $missing);
                $changed = true;
                $i += count($missing);
            }
            $insideVhost = false;
        }
    }

    if ($changed) {
        // Create backup of original httpd-vhosts.conf with timestamp
        $backupPath = $vhostsPath . '_' . date('Ymd_His');
        if (!@copy($vhostsPath, $backupPath)) {
            $error = 'Failed to create backup of httpd-vhosts.conf.';
            $status = 'error';
            return false;
        }
        $result = @file_put_contents($vhostsPath, implode('', $lines));
        if ($result === false) {
            $error  = 'Failed to write updated httpd-vhosts.conf.';
            $status = 'error';
            return false;
        }
        $status = 'updated';
        error_log("servercontrol_apacheconfig.php: httpd-vhosts.conf updated at $vhostsPath (backup created at $backupPath)");
    } else {
        $status = 'unchanged';
        error_log("servercontrol_apacheconfig.php: httpd-vhosts.conf unchanged at $vhostsPath");
    }

    return true;
}
?>