<?php
/**
 * HTTP Proxy Script
 * 
 * This script acts as a simple HTTP proxy that forwards requests to a target URL
 * and returns the response to the client. It supports both streaming and 
 * non-streaming responses.
 * 
 * Usage:
 *   GET:  ?rurl=https://example.com/api/endpoint
 *   POST: ?rurl=https://example.com/api/endpoint (with POST body)
 * 
 * Features:
 * - HTTP GET and POST support
 * - Header forwarding
 * - Body forwarding for POST requests
 * - Streaming and non-streaming response support
 * - CORS support
 * - Transaction logging
 */

// Set timezone
date_default_timezone_set("UTC");

// Include proxy logger
require_once __DIR__ . '/proxy_logger.php';

// Get datadir from environment or use default
$datadir = isset($datadir) ? $datadir : '/var/ozeki/data';

// Load configurations
include_once("$servicesdir/aigate/setup.php");

// Get the target URL from the rurl parameter
$rurl = isset($_GET['rurl']) ? $_GET['rurl'] : '';

// Validate the rurl parameter
if (empty($rurl)) {
    http_response_code(400);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => 'Missing required parameter: rurl',
        'usage' => 'GET/POST ?rurl=<target_url>'
    ]);
    exit;
}

//error_log("proxy.php: Proxy request for ".$rurl);

// Validate URL format
$validatedUrl = filter_var($rurl, FILTER_VALIDATE_URL);
if ($validatedUrl === false) {
    http_response_code(400);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => 'Invalid URL format: ' . $rurl
    ]);
    exit;
}

// Only allow HTTP and HTTPS protocols
$scheme = parse_url($validatedUrl, PHP_URL_SCHEME);
if (!in_array($scheme, ['http', 'https'])) {
    http_response_code(400);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => 'Only HTTP and HTTPS protocols are allowed'
    ]);
    exit;
}

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

// Handle CORS preflight requests
if ($method === 'OPTIONS') {
    // Allow CORS
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept, X-Requested-With, User-Agent');
    header('Access-Control-Max-Age: 86400');
    http_response_code(200);
    exit;
}

// Set CORS headers for all responses
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept, X-Requested-With, User-Agent');

// Get request body for POST requests
$requestBody = '';
if ($method === 'POST') {
    $requestBody = file_get_contents('php://input');
}

// Function to get all incoming headers
if (!function_exists('getallheaders')) {
    function getallheaders() {
        $headers = [];
        foreach ($_SERVER as $name => $value) {
            // Skip internal server headers
            if (strpos($name, 'HTTP_') === 0) {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                $headers[$key] = $value;
            } elseif (in_array($name, ['CONTENT_TYPE', 'CONTENT_LENGTH'])) {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', $name))));
                $headers[$key] = $value;
            }
        }
        return $headers;
    }
}

// Get all incoming headers
$incomingHeaders = getallheaders();

//error_log("proxy.php: Incoming headers ".json_encode($incomingHeaders));

// Build headers to forward to the target
$forwardHeaders = [];
$hopByHopHeaders = [
    'Connection',
    'Keep-Alive',
    'Proxy-Authenticate',
    'Proxy-Authorization',
    'TE',
    'Trailers',
    'Transfer-Encoding',
    'Accept-Encoding',
    'Upgrade',
    'Host' // We'll set this separately based on the target URL
];

foreach ($incomingHeaders as $key => $value) {
    $keyLower = strtolower($key);
    // Skip hop-by-hop headers and host header
    if (!in_array($keyLower, array_map('strtolower', $hopByHopHeaders)) && $keyLower !== 'host') {
        $forwardHeaders[] = "$key: $value";
    }
}
$forwardHeaders[] = 'Accept-Encoding: gzip'; //több féle tömörítés létezik, csak a gzip-et támogatjuk

// Add Content-Type and Content-Length for POST requests if not present
if ($method === 'POST') {
    if (!empty($requestBody)) {
        $contentType = isset($incomingHeaders['Content-Type']) ? $incomingHeaders['Content-Type'] : 'application/x-www-form-urlencoded';
    }
}

// Check if streaming is requested
$isStreaming = isset($_GET['stream']) && $_GET['stream'] === 'true';

// Initialize cURL
$ch = curl_init();

// Set cURL options based on request method
if ($method === 'GET') {
    curl_setopt($ch, CURLOPT_URL, $validatedUrl);
    curl_setopt($ch, CURLOPT_HTTPGET, true);
} else {
    curl_setopt($ch, CURLOPT_URL, $validatedUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $requestBody);
}

// Set headers
curl_setopt($ch, CURLOPT_HTTPHEADER, $forwardHeaders);

// Set other cURL options
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HEADER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
curl_setopt($ch, CURLOPT_TIMEOUT, 60);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
curl_setopt($ch, CURLOPT_USERAGENT, isset($incomingHeaders['User-Agent']) ? $incomingHeaders['User-Agent'] : 'Proxy/1.0');

// Prepare request data for logging
$requestData = [
    'url' => $validatedUrl,
    'method' => $method,
    'headers' => $incomingHeaders,
    'body' => $method === 'POST' ? $requestBody : null,
    'client' => proxyGetClientIp()
];

//error_log("proxy.php: Request data ".json_encode($requestData));

// Handle streaming vs non-streaming
if ($isStreaming) {
    // For streaming, we need to output data as it comes
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, false);
    curl_setopt($ch, CURLOPT_WRITEFUNCTION, function($ch, $data) {
        echo $data;
        if (ob_get_level() > 0) {
            ob_flush();
        }
        flush();
        return strlen($data);
    });
}

// Execute the request - start timing
$startTime = microtime(true);

if ($isStreaming) {
    // For streaming, we need to set proper headers first
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');
    header('X-Accel-Buffering: no'); // Disable nginx buffering
    
    // Execute the streaming request
    $result = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Calculate duration
    $duration = microtime(true) - $startTime;
    
    // Prepare response data for logging
    $responseData = [
        'http_code' => $httpCode,
        'body' => $result,
        'error' => $curlError ?: null
    ];
    
    //error_log("proxy.php: Logging streaming transcation ".json_encode($responseData));
    // Log the transaction
    proxyLogTransaction($requestData, $responseData, '', [], $duration, $curlError ?: null);
    
    if ($curlError) {
        // Send error as SSE format if streaming
        echo "data: " . json_encode(['error' => 'Failed to connect to target server: ' . $curlError]) . "\n\n";
        flush();
    }
} else {
    // Non-streaming request
    header('Cache-Control: no-cache');
    header('Connection: close');
    $result = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Calculate duration
    $duration = microtime(true) - $startTime;
    
    if ($curlError) {
        // Prepare response data for logging
        $responseData = [
            'http_code' => $httpCode,
            'body' => null,
            'error' => $curlError
        ];
        
        //error_log("proxy.php: Error ".json_encode($responseData));
        // Log the transaction
        proxyLogTransaction($requestData, $responseData, '', [], $duration, $curlError);
        
        http_response_code(502);
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Failed to connect to target server: ' . $curlError]);
        exit;
    }
    
    // Separate headers and body
     ob_start();
     var_dump($result);
     $resp = ob_get_contents();
     ob_end_clean();

    $responseHeaders = substr($result, 0, $headerSize);
    $responseBody = substr($result, $headerSize);
    if (!isset($responseBody)) $responseBody="";
    
    // Check if response is gzip compressed and decompress if needed
    $isGzipEncoded = false;
    $uncompressedBody = $responseBody;
    
    // Parse response headers to check for Content-Encoding: gzip
    $responseHeaderLines = explode("\r\n", $responseHeaders);
    foreach ($responseHeaderLines as $line) {
        if (!empty($line) && strpos($line, 'HTTP/') === false) {
            $headerParts = explode(':', $line, 2);
            if (count($headerParts) === 2) {
                $headerName = trim($headerParts[0]);
                $headerValue = trim($headerParts[1]);
                if (strtolower($headerName) === 'content-encoding' && strtolower($headerValue) === 'gzip') {
                    $isGzipEncoded = true;
                    break;
                }
            }
        }
    }
    
    // Decompress gzip response if needed
    if ($isGzipEncoded && !empty($responseBody)) {
        $uncompressedBody = gzdecode($responseBody);
        if ($uncompressedBody === false) {
            //error_log("proxy.php: Failed to decompress gzip response");
            $uncompressedBody = $responseBody; // Fall back to original
        } else {
           // error_log("proxy.php: Successfully decompressed gzip response");
        }
    }
    
    // Prepare response data for logging (use uncompressed body if gzip was detected)
    $responseHeaderLines = $lines = preg_split('/\r\n|\r|\n/', $responseHeaders);
    $responseHeaderArray = [];
    foreach($responseHeaderLines as $line)
    {
        if (empty(trim($line))) continue;
        if (strpos($line,":")<2) continue;
        list($key,$value) = explode(': ',$line,2);
        $responseHeaderArray[$key] = $value;
    }

    $responseData = [
        'http_code' => $httpCode,
        'headers' => $responseHeaderArray,
        'body' => $uncompressedBody
    ];

    // Log the transaction
    proxyLogTransaction($requestData, $responseData, '', [], $duration, null, $rurl);
    
    // Parse and forward response headers
    $responseHeaderLines = preg_split('/\r\n|\r|\n/', $responseHeaders);
    foreach ($responseHeaderLines as $line) {
        if (!empty($line) && strpos($line, 'HTTP/') === false) {
            // Skip hop-by-hop headers for the response
            $headerParts = explode(':', $line, 2);
            if (count($headerParts) === 2) {
                $headerName = trim($headerParts[0]);
                $headerValue = trim($headerParts[1]);
                
                $headerNameLower = strtolower($headerName);
                if (!in_array($headerNameLower, [
                    'transfer-encoding', 
                    'content-encoding', 
                    'connection', 
                    'keep-alive', 
                    'proxy-authenticate', 
                    'proxy-authorization', 
                    'te', 
                    'trailers', 
                    'upgrade'])) {
                    header("$headerName: $headerValue");
                }
            }
        }
    }
    
    // Set the HTTP status code
    http_response_code($httpCode);
    
    // Output the response body (use uncompressed body if gzip was detected)
    echo $uncompressedBody;
}

exit;
?>