<?php

// https://platform.openai.com/docs/api-reference/chat/create
// https://platform.claude.com/docs/en/api/messages/create

/**
 * Converts an Anthropic-style messages request to OpenAI-style format.
 *
 * Handles key differences including:
 * - System messages (Anthropic uses separate 'system' parameter)
 * - Image content blocks (different base64 encoding formats)
 * - Tool use and tool results
 * - Role naming conventions
 * - Metadata (custom key-value pairs for tracking/correlation)
 *
 * @param array $anthropicRequest The Anthropic-style request array
 * @return array The converted OpenAI-style request array
 */
function transformAnthropicMessageReqToOpenAiChatCompletionReq(object $anthropicRequest): object
{
    $openaiRequest = new stdClass();

    // Initialize messages array
    $openaiRequest->messages = [];

    // Handle system message
    if (isset($anthropicRequest->system)) {
        $openaiRequest->messages[] = anthropicMessageReqToOpenAiChatCompletionReq_convertSystemMessage($anthropicRequest->system);
    }

    // Convert user and assistant messages and merge with system message if present
    if (isset($anthropicRequest->messages) && is_array($anthropicRequest->messages)) {
        $openaiRequest->messages = array_merge($openaiRequest->messages, anthropicMessageReqToOpenAiChatCompletionReq_convertMessages($anthropicRequest->messages));
    }

    // Convert model name if present
    if (isset($anthropicRequest->model)) {
        $openaiRequest->model = $anthropicRequest->model;
    }

    // Convert max_tokens to max_completion_tokens
    if (isset($anthropicRequest->max_tokens)) {
        $openaiRequest->max_completion_tokens = $anthropicRequest->max_tokens;
    }

    // Convert metadata if present
    if (isset($anthropicRequest->metadata) && is_object($anthropicRequest->metadata)) {
        $openaiRequest->metadata = $anthropicRequest->metadata;
    }

    // Convert stop sequences
    if (isset($anthropicRequest->stop_sequences)) {
        $openaiRequest->stop = $anthropicRequest->stop_sequences;
    }

    // Handle streaming
    if (isset($anthropicRequest->stream)) {
        $openaiRequest->stream = $anthropicRequest->stream;

        // OpenAI uses stream_options for additional streaming configuration
        if ($anthropicRequest->stream === true) {
            $openaiRequest->stream_options = new stdClass();
            $openaiRequest->stream_options->include_usage = true;
        }
    }

    // Convert temperature
    if (isset($anthropicRequest->temperature)) {
        $openaiRequest->temperature = $anthropicRequest->temperature;
    }

    // Convert tool_choice if present
    if (isset($anthropicRequest->tool_choice)) {
        $openaiRequest->tool_choice = anthropicMessageReqToOpenAiChatCompletionReq_convertToolChoice($anthropicRequest->tool_choice);
    }

    // Convert tools if present
    if (isset($anthropicRequest->tools)) {
        $openaiRequest->tools = anthropicMessageReqToOpenAiChatCompletionReq_convertTools($anthropicRequest->tools);
    }

    // Convert top_p
    if (isset($anthropicRequest->top_p)) {
        $openaiRequest->top_p = $anthropicRequest->top_p;
    }

    return $openaiRequest;
}

/**
 * Converts Anthropic system message to OpenAI format.
 *
 * Handles both string system messages and arrays of system message blocks.
 * If an array is provided, extracts 'text' field from each block and joins with double newlines.
 *
 * @param array|string $system The Anthropic system message content
 * @return object The converted OpenAI system message
 */
function anthropicMessageReqToOpenAiChatCompletionReq_convertSystemMessage($system): object
{
    $systemContent = is_array($system)
        ? implode("\n\n", array_column($system, 'text'))
        : $system;

    return (object)[
        'role' => 'system',
        'content' => $systemContent
    ];
}

/**
 * Converts an array of Anthropic messages to OpenAI format.
 *
 * Extracts tool_calls from content blocks and places them at message level.
 * Tool results are converted to separate OpenAI tool messages.
 *
 * @param array $messages The Anthropic messages array
 * @return array Array of converted OpenAI messages (may contain multiple if tool_results are present)
 */
function anthropicMessageReqToOpenAiChatCompletionReq_convertMessages(array $messages): array
{
    $openaiMessages = [];

    foreach ($messages as $message) {
        // Skip message blocks without required fields
        if (!isset($message->role) || !isset($message->content)) {
            continue;
        }

        $openaiMessage = new stdClass();
        $openaiMessage->role = ($message->role === 'user') ? 'user' : 'assistant';

        // Handle string content (simple text messages)
        if (is_string($message->content)) {
            $openaiMessage->content = $message->content;
            $openaiMessages[] = $openaiMessage;
            continue;
        }

        // Skip if content is not an array of content blocks
        if (!is_array($message->content)) {
            continue;
        }

        // Convert content blocks (text, images, tool_use, tool_result)
        $contentResult = anthropicMessageReqToOpenAiChatCompletionReq_convertContent($message->content);

        // Set content if present
        if (!empty($contentResult['content'])) {
            $openaiMessage->content = $contentResult['content'];
        }

        // Add tool_calls at message level if present
        if (!empty($contentResult['tool_calls'])) {
            $openaiMessage->tool_calls = $contentResult['tool_calls'];
        }

        // Only add main message if it has content or tool_calls
        if (isset($openaiMessage->content) || isset($openaiMessage->tool_calls)) {
            $openaiMessages[] = $openaiMessage;
        }

        // Add tool result messages separately
        if (!empty($contentResult['tool_messages'])) {
            foreach ($contentResult['tool_messages'] as $toolMessage) {
                $openaiMessages[] = $toolMessage;
            }
        }
    }

    return $openaiMessages;
}

/**
 * Converts Anthropic content blocks to OpenAI format.
 *
 * Processes content blocks and organizes them:
 * - Text and image blocks are converted to 'content' array
 * - Tool use blocks are extracted for message-level 'tool_calls' property
 * - Tool result blocks are converted to separate OpenAI tool messages
 *
 * @param array $contentBlocks Array of Anthropic content blocks
 * @return array Array with 'content', 'tool_calls', and 'tool_messages' keys
 */
function anthropicMessageReqToOpenAiChatCompletionReq_convertContent(array $contentBlocks): array
{
    $openaiContent = [];
    $openaiToolCalls = [];
    $openaiToolMessages = [];

    foreach ($contentBlocks as $block) {
        if (!is_object($block) || !isset($block->type)) {
            continue;
        }

        switch ($block->type) {
            case 'text':
                if (!isset($block->text)) break;

                $openaiContent[] = (object)[
                    'type' => 'text',
                    'text' => $block->text,
                ];
                break;

            case 'image':
                if (!isset($block->source) || !is_object($block->source) || !isset($block->source->type)) {
                    break;
                }

                $imageUrl = ($block->source->type === 'base64')
                    ? ("data:{$block->source->media_type};base64,{$block->source->data}")
                    : $block->source->url;

                $openaiContent[] = (object)[
                    'type' => 'image_url',
                    'image_url' => (object)['url' => $imageUrl],
                ];
                break;

            case 'tool_use':
                // Extract tool_use for tool_calls at message level
                if (!isset($block->id) || !isset($block->name) || !isset($block->input)) break;

                $openaiToolCalls[] = (object)[
                    'id' => $block->id,
                    'type' => 'function',
                    'function' => (object)[
                        'name' => $block->name,
                        'arguments' => is_string($block->input) ? $block->input : json_encode($block->input),
                    ],
                ];
                break;

            case 'tool_result':
                // Convert tool results to OpenAI tool messages
                if (!isset($block->tool_use_id) || !isset($block->content)) break;

                $openaiToolMessages[] = (object)[
                    'role' => 'tool',
                    'tool_call_id' => $block->tool_use_id,
                    'content' => $block->content,
                ];
                break;
        }
    }

    return [
        'content' => $openaiContent,
        'tool_calls' => $openaiToolCalls,
        'tool_messages' => $openaiToolMessages,
    ];
}

/**
 * Converts Anthropic tool_choice to OpenAI format.
 *
 * Handles string tool_choice values ('auto', 'any', 'none') and object definitions.
 * Converts 'any' to OpenAI's 'required'. Other string values pass through unchanged.
 * Object definitions with type 'tool' are converted to OpenAI's function format.
 *
 * @param array|string $toolChoice The Anthropic tool_choice value
 * @return array|string The converted OpenAI tool_choice value
 */
function anthropicMessageReqToOpenAiChatCompletionReq_convertToolChoice($toolChoice)
{
    if (is_string($toolChoice)) {
        // Convert Anthropic 'any' to OpenAI 'required'; 'auto' and 'none' remain unchanged
        return $toolChoice === 'any' ? 'required' : $toolChoice;
    }

    if (is_object($toolChoice) && isset($toolChoice->type)) {
        if ($toolChoice->type === 'tool' && isset($toolChoice->name)) {
            // Convert tool object to OpenAI function format
            return (object)[
                'type' => 'function',
                'function' => (object)['name' => $toolChoice->name],
            ];
        }
    }

    return 'auto';
}

/**
 * Converts Anthropic tools to OpenAI format.
 *
 * Wraps each tool in OpenAI's function format:
 * - Converts 'input_schema' to 'parameters'
 * - Provides empty string default for missing 'description'
 * - Provides default empty object schema if 'input_schema' is missing
 *
 * @param array $tools Array of Anthropic tool definitions
 * @return array Array of OpenAI tool definitions
 */
function anthropicMessageReqToOpenAiChatCompletionReq_convertTools(array $tools): array
{
    $openaiTools = [];

    foreach ($tools as $tool) {
        if (!is_object($tool) || !isset($tool->name)) {
            continue;
        }

        $openaiTools[] = (object)[
            'type' => 'function',
            'function' => (object)[
                'name' => $tool->name,
                'description' => $tool->description ?? '',
                'parameters' => $tool->input_schema ?? (object)['type' => 'object', 'properties' => (object)[]],
            ],
        ];
    }

    return $openaiTools;
}
?>