<?php
include_once("$dirprefix/ozekiservices/aigate/api/gateway/anthropic/anthropic_inc_fromopenai.php");

//https://platform.openai.com/docs/api-reference/chat-streaming
//https://platform.openai.com/docs/guides/function-calling?api-mode=chat#streaming
//https://platform.claude.com/docs/en/build-with-claude/streaming
//https://platform.claude.com/docs/en/api/errors

function transformOpenAiStreamingEventToAnthropic(string $eventChunk, array &$state, array $options = []): array
{
    $events = [];

    if (isset($state['error_occurred']) && $state['error_occurred']) {
        return $events;
    }

    if (!isset($state['buffer'])) {
        $state['buffer'] = '';
    }
    $state['buffer'] .= $eventChunk;
    if (!str_contains($state['buffer'], "\n\n")) {
        return $events;
    }

    while (($pos = strpos($state['buffer'], "\n\n")) !== false) {
        $event = substr($state['buffer'], 0, $pos);
        $state['buffer'] = substr($state['buffer'], $pos + 2);

        if (!str_starts_with($event, 'data: ')) {
            $state['error_occurred'] = true;
            $events[] = openAiStreamingEventToAnthropic_createEventError('api_error', 'Invalid streaming event format from OpenAI.');
            return $events;
        }

        $jsonData = substr($event, 6);
        if ($jsonData === '[DONE]') {
            $events = array_merge($events, openAiStreamingEventToAnthropic_handleStreamDone($state));
            return $events;
        }

        $events = array_merge($events, openAiStreamingEventToAnthropic_handleJsonData($state, $options, $jsonData));
    }

    return $events;
}

function openAiStreamingEventToAnthropic_handleStreamDone(array &$state): array
{
    $events = [];

    $events[] = openAiStreamingEventToAnthropic_createEvent((object)[
        'type' => 'message_delta',
        'delta' => (object)[
            'stop_reason' => $state['stop_reason'] ?? 'end_turn',
            'stop_sequence' => null,
        ],
        'usage' => (object)[
            'input_tokens' => $state['input_tokens'] ?? 0,
            'output_tokens' => $state['output_tokens'] ?? 0,
        ],
    ]);

    $events[] = openAiStreamingEventToAnthropic_createEvent((object)[
        'type' => 'message_stop',
    ]);

    $events[] = "data: [DONE]\n\n";

    return $events;
}

function openAiStreamingEventToAnthropic_handleJsonData(array &$state, array $options, string $jsonData): array
{
    $events = [];

    $data = json_decode($jsonData, false);
    if (json_last_error() !== JSON_ERROR_NONE) {
        $state['error_occurred'] = true;
        $events[] = openAiStreamingEventToAnthropic_createEventError('api_error', 'Failed to decode JSON data from OpenAI streaming event.');
        return $events;
    }

    if (!isset($state['message_id']) && isset($data->id) && !empty($data->id)) {
        $state['message_id'] = $data->id;
    }

    if (!isset($state['model']) && isset($data->model) && !empty($data->model)) {
        $state['model'] = $data->model;
    }

    if (isset($data->usage)) {
        $state['input_tokens'] = $data->usage->prompt_tokens ?? 0;
        $state['output_tokens'] = $data->usage->completion_tokens ?? 0;
    }

    $choice = $data->choices[0] ?? null;
    if (empty($choice)) {
        return $events;
    }

    if (!isset($state['message_started']) || !$state['message_started']) {
        $events[] = openAiStreamingEventToAnthropic_createEventMessageStart($state);
        $state['message_started'] = true;
        $state['next_content_block_index'] = 0;
    }

    $delta = $choice->delta ?? null;

    if (isset($delta->content) && !empty($delta->content)) {
        $events = array_merge(
            $events,
            openAiStreamingEventToAnthropic_handleDeltaContent($state, $delta->content)
        );
    }

    //note reasoning_content
    if (isset($options['thinking_enabled']) && $options['thinking_enabled'] && isset($delta->reasoning) && !empty($delta->reasoning)) {
        $events = array_merge(
            $events,
            openAiStreamingEventToAnthropic_handleDeltaReasoning($state, $delta->reasoning)
        );
    }

    if (isset($delta->tool_calls) && !empty($delta->tool_calls)) {
        $events = array_merge(
            $events,
            openAiStreamingEventToAnthropic_handleDeltaToolCalls($state, $delta->tool_calls)
        );
    }

    if (isset($choice->finish_reason) && !empty($choice->finish_reason)) {
        $events = array_merge(
            $events,
            openAiStreamingEventToAnthropic_handleFinishReason($state, $choice->finish_reason)
        );
    }

    return $events;
}

function openAiStreamingEventToAnthropic_handleDeltaReasoning(array &$state, string $reasoning): array
{
    $events = [];

    if (!isset($state['thinking_content_block_index']) || $state['thinking_content_block_index'] < 0) {
        $state['thinking_content_block_index'] = $state['next_content_block_index']++;
        $data = (object)[
            'type' => 'content_block_start',
            'index' => $state['thinking_content_block_index'],
            'content_block' => (object)[
                'type' => 'thinking',
                'thinking' => '',
            ],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
    }

    if (!empty($reasoning)) {
        $data = (object)[
            'type' => 'content_block_delta',
            'index' => $state['thinking_content_block_index'],
            'delta' => (object)[
                'type' => 'thinking_delta',
                'thinking' => $reasoning,
            ],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
    }

    return $events;
}

function openAiStreamingEventToAnthropic_handleDeltaContent(array &$state, string $content): array
{
    $events = [];

    $events = array_merge($events, openAiStreamingEventToAnthropic_stopThinkingContentBlock($state));

    if (!isset($state['text_content_block_index']) || $state['text_content_block_index'] < 0) {
        $state['text_content_block_index'] = $state['next_content_block_index']++;
        $data = (object)[
            'type' => 'content_block_start',
            'index' => $state['text_content_block_index'],
            'content_block' => (object)[
                'type' => 'text',
                'text' => '',
            ],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
    }

    if (!empty($content)) {
        $data = (object)[
            'type' => 'content_block_delta',
            'index' => $state['text_content_block_index'],
            'delta' => (object)[
                'type' => 'text_delta',
                'text' => $content,
            ],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
    }

    return $events;
}

function openAiStreamingEventToAnthropic_handleDeltaToolCalls(array &$state, array $tool_calls): array
{
    $events = [];

    if (!isset($state['tool_calls_content_block_index_map'])) {
        $state['tool_calls_content_block_index_map'] = [];
    }

    if (!empty($tool_calls)) {
        //note
        $events = array_merge($events, openAiStreamingEventToAnthropic_stopThinkingContentBlock($state));
        $events = array_merge($events, openAiStreamingEventToAnthropic_stopTextContentBlock($state));
    }

    foreach ($tool_calls as $tool_call) {
        $toolCallIndex = $tool_call->index;

        if (!isset($state['tool_calls_content_block_index_map'][$toolCallIndex])) {
            $blockIndex = $state['next_content_block_index']++;
            $state['tool_calls_content_block_index_map'][$toolCallIndex] = $blockIndex;

            $data = (object)[
                'type' => 'content_block_start',
                'index' => $blockIndex,
                'content_block' => (object)[
                    'type' => 'tool_use',
                    'id' => $tool_call->id ?? 'toolu_' . uniqid(),
                    'name' => $tool_call->function->name ?? '',
                    'input' => (object)[],
                ],
            ];
            $events[] = openAiStreamingEventToAnthropic_createEvent($data);
        }

        $functionArguments = $tool_call->function->arguments ?? '';
        if (!empty($functionArguments)) {
            $blockIndex = $state['tool_calls_content_block_index_map'][$toolCallIndex];

            $data = (object)[
                'type' => 'content_block_delta',
                'index' => $blockIndex,
                'delta' => (object)[
                    'type' => 'input_json_delta',
                    'partial_json' => $functionArguments,
                ],
            ];
            $events[] = openAiStreamingEventToAnthropic_createEvent($data);
        }
    }

    return $events;
}

function openAiStreamingEventToAnthropic_handleFinishReason(array &$state, string $finishReason): array
{
    $state['stop_reason'] = openAiChatCompletionRespToAnthropicMessageResp_mapFinishReason($finishReason);

    $events = [];
    $events = array_merge($events, openAiStreamingEventToAnthropic_stopThinkingContentBlock($state));
    $events = array_merge($events, openAiStreamingEventToAnthropic_stopTextContentBlock($state));
    $events = array_merge($events, openAiStreamingEventToAnthropic_stopToolUseContentBlock($state));

    return $events;
}

function openAiStreamingEventToAnthropic_createEventMessageStart(array $state): string
{
    $data = (object)[
        'type' => 'message_start',
        'message' => (object)[
            'id' => $state['message_id'] ?? 'msg_' . uniqid(),
            'type' => 'message',
            'role' => 'assistant',
            'content' => [],
            'model' => $state['model'] ?? 'unknown',
            'stop_reason' => null,
            'stop_sequence' => null,
            'usage' => (object)[
                'input_tokens' => $state['input_tokens'] ?? 0,
                'output_tokens' => $state['output_tokens'] ?? 0,
            ],
        ],
    ];

    return openAiStreamingEventToAnthropic_createEvent($data);
}

function openAiStreamingEventToAnthropic_createEventError(string $type, string $message, string $requestId = ''): string
{
    $data = (object)[
        'type' => 'error',
        'error' => (object)[
            'type' => $type,
            'message' => $message,
        ],
        'request_id' => $requestId,
    ];

    return openAiStreamingEventToAnthropic_createEvent($data);
}

function openAiStreamingEventToAnthropic_stopThinkingContentBlock(array &$state): array
{
    $events = [];

    if (isset($state['thinking_content_block_index']) && $state['thinking_content_block_index'] > -1) {
        //todo send signature_delta before content_block_stop
        $data = (object)[
            'type' => 'content_block_stop',
            'index' => $state['thinking_content_block_index'],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
        $state['thinking_content_block_index'] = -1;
    }

    return $events;
}

function openAiStreamingEventToAnthropic_stopTextContentBlock(array &$state): array
{
    $events = [];

    if (isset($state['text_content_block_index']) && $state['text_content_block_index'] > -1) {
        $data = (object)[
            'type' => 'content_block_stop',
            'index' => $state['text_content_block_index'],
        ];
        $events[] = openAiStreamingEventToAnthropic_createEvent($data);
        $state['text_content_block_index'] = -1;
    }

    return $events;
}

function openAiStreamingEventToAnthropic_stopToolUseContentBlock(array &$state): array
{
    $events = [];

    if (isset($state['tool_calls_content_block_index_map'])) {
        foreach ($state['tool_calls_content_block_index_map'] as $toolCallIndex => $blockIndex) {
            $data = (object)[
                'type' => 'content_block_stop',
                'index' => $blockIndex,
            ];
            $events[] = openAiStreamingEventToAnthropic_createEvent($data);
            unset($state['tool_calls_content_block_index_map'][$toolCallIndex]);
        }
    }

    return $events;
}

function openAiStreamingEventToAnthropic_createEvent(object $data): string
{
    $type = $data->type;
    return "event: {$type}\ndata: " . json_encode($data) . "\n\n";
}
?>