<?php
//https://platform.openai.com/docs/api-reference/chat/create
//https://platform.openai.com/docs/guides/error-codes
//https://platform.claude.com/docs/en/api/messages/create
//https://platform.claude.com/docs/en/api/errors

/**
 * Converts an OpenAI-style messages response to Anthropic-style format.
 *
 * Handles key differences including:
 * - Response structure (choices array vs single response)
 * - Role naming (assistant vs assistant)
 * - Content blocks format
 * - Tool calls vs tool use blocks
 * - Usage statistics format
 * - Finish reasons
 * - Reasoning tokens (OpenAI o1 models)
 *
 * @param object $openaiResponse The OpenAI-style response object
 * @param array $options Options for conversion (e.g. thinking_enabled)
 * @param int $httpCode Reference to HTTP status code variable to set
 * @return object The converted Anthropic-style response object
 */
function transformOpenAiChatCompletionRespToAnthropicMessageResp(object $openaiResponse, $options, &$httpCode): object
{
    // Check for error response
    if (openAiChatCompletionRespToAnthropicMessageResp_isError($openaiResponse)) {
        return openAiChatCompletionRespToAnthropicMessageResp_convertError($openaiResponse, $httpCode);
    }

    // Return error if choices array is empty
    if (!isset($openaiResponse->choices) || empty($openaiResponse->choices)) {
        $httpCode = 500;
        return openAiChatCompletionRespToAnthropicMessageResp_createError('api_error', 'No choices in OpenAI response.');
    }
    
    $anthropicResponse = (object)[
        'id' => $openaiResponse->id ?? 'msg_' . bin2hex(random_bytes(16)),
        'model' => $openaiResponse->model ?? 'unknown',
        'role' => 'assistant',
        'stop_sequence' => null,
        'type' => 'message',
    ];
    
    // Convert content from first choice
    $choice = $openaiResponse->choices[0];

    // Convert content blocks
    if (isset($choice->message)) {
        $anthropicResponse->content = openAiChatCompletionRespToAnthropicMessageResp_createContent($choice->message, $options);
    }

    // Convert finish reason
    if (isset($choice->finish_reason)) {
        $anthropicResponse->stop_reason = openAiChatCompletionRespToAnthropicMessageResp_mapFinishReason($choice->finish_reason);
    }

    // Convert usage statistics
    if (isset($openaiResponse->usage)) {
        $anthropicResponse->usage = openAiChatCompletionRespToAnthropicMessageResp_convertUsage($openaiResponse->usage);
    }

    $httpCode = 200;
    return $anthropicResponse;
}

/**
 * Converts OpenAI message content to Anthropic content blocks format.
 *
 * @param object $message The OpenAI message object
 * @param array $options Options for conversion (e.g. thinking_enabled)
 * @return array Array of Anthropic content blocks
 */
function openAiChatCompletionRespToAnthropicMessageResp_createContent(object $message, array $options): array
{
    $contentBlocks = [];
    
    // Handle refusal (OpenAI-specific field) - appears before main content
    if (isset($message->refusal) && !empty($message->refusal)) {
        $contentBlocks[] = (object)[
            'type' => 'text',
            'text' => $message->refusal
        ];
    }
    
    // Convert text and image content
    if (isset($message->content)) {
        $contentBlocks = array_merge($contentBlocks, openAiChatCompletionRespToAnthropicMessageResp_convertContent($message->content));
    }

    // Handle reasoning content (OpenAI o1 models)
    if (isset($options['thinking_enabled']) && $options['thinking_enabled'] && (isset($message->reasoning) || isset($message->reasoning_content))) {
        $contentBlocks = array_merge($contentBlocks, openAiChatCompletionRespToAnthropicMessageResp_convertReasoning($message));
    }
    
    // Handle tool calls
    if (isset($message->tool_calls) && is_array($message->tool_calls)) {
        $contentBlocks = array_merge($contentBlocks, openAiChatCompletionRespToAnthropicMessageResp_convertToolCalls($message->tool_calls));
    }
    
    return $contentBlocks;
}

/**
 * Converts OpenAI message content (text and images) to Anthropic content blocks.
 *
 * Handles both string content and arrays of content parts (text and image_url).
 *
 * @param array|string $content The OpenAI message content
 * @return array Array of Anthropic content blocks (text and image)
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertContent($content): array
{
    $contentBlocks = [];
    
    // Handle string content
    if (is_string($content) && !empty($content)) {
        $contentBlocks[] = (object)[
            'type' => 'text',
            'text' => $content
        ];
        return $contentBlocks;
    }

    // Invalid content format
    if (!is_array($content)) {
        return $contentBlocks;
    }
    
    // Handle array of content parts
    foreach ($content as $part) {
        switch ($part->type) {
            case 'text':
                $contentBlocks[] = (object)[
                    'type' => 'text',
                    'text' => $part->text
                ];
                break;
            
            case 'image_url':
                $contentBlocks[] = openAiChatCompletionRespToAnthropicMessageResp_convertContentImage($part);
                break;
        }
    }
    
    return $contentBlocks;
}

/**
 * Converts OpenAI image content to Anthropic format.
 *
 * @param object $imagePart The OpenAI image_url content part
 * @return object Anthropic image content block
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertContentImage(object $imagePart): object
{
    $imageUrl = $imagePart->image_url->url ?? '';
    
    // Check if it's a base64 data URL
    if (preg_match('/^data:(image\/[a-z]+);base64,(.+)$/i', $imageUrl, $matches)) {
        return (object)[
            'type' => 'image',
            'source' => (object)[
                'type' => 'base64',
                'media_type' => $matches[1],
                'data' => $matches[2]
            ]
        ];
    }
    
    // Handle regular URL
    return (object)[
        'type' => 'image',
        'source' => (object)[
            'type' => 'url',
            'url' => $imageUrl
        ]
    ];
}

/**
 * Converts OpenAI reasoning content to Anthropic format.
 *
 * Handles reasoning fields from OpenAI o1 models by converting them to text content blocks.
 * Checks for both 'reasoning' and 'reasoning_content' fields.
 *
 * @param object $message The OpenAI message object
 * @return array Array of content blocks (may be empty if no reasoning present)
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertReasoning(object $message): array
{
    $contentBlocks = [];
    
    // Check for reasoning field (OpenAI o1 models)
    if (isset($message->reasoning) && !empty($message->reasoning)) {
        $contentBlocks[] = (object)[
            'type' => 'text',
            'text' => $message->reasoning
        ];
    } elseif (isset($message->reasoning_content) && !empty($message->reasoning_content)) {
        $contentBlocks[] = (object)[
            'type' => 'text',
            'text' => $message->reasoning_content
        ];
    }
    
    return $contentBlocks;
}

/**
 * Converts OpenAI tool calls to Anthropic tool_use content blocks.
 *
 * Extracts function tool calls and converts them to Anthropic's tool_use format.
 *
 * @param array $toolCalls The OpenAI tool_calls array
 * @return array Array of Anthropic tool_use content blocks
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertToolCalls(array $toolCalls): array
{
    $toolUseBlocks = [];
    
    foreach ($toolCalls as $toolCall) {
        if ($toolCall->type === 'function') {
            $toolUseBlocks[] = (object)[
                'type' => 'tool_use',
                'id' => $toolCall->id,
                'name' => $toolCall->function->name,
                'input' => json_decode($toolCall->function->arguments, false) ?? (object)[]
            ];
        }
    }
    
    return $toolUseBlocks;
}

/**
 * Converts OpenAI usage statistics to Anthropic format.
 *
 * Maps OpenAI token counts to Anthropic equivalents:
 * - prompt_tokens → input_tokens
 * - completion_tokens → output_tokens
 * - Handles cache-related token counts if present
 *
 * @param array $usage The OpenAI usage statistics
 * @return array The converted Anthropic usage statistics
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertUsage(object $usage): object
{
    $anthropicUsage = (object)[
        'input_tokens' => $usage->prompt_tokens ?? 0,
        'output_tokens' => $usage->completion_tokens ?? 0
    ];
    
    // Add cache creation tokens if applicable
    if (isset($usage->prompt_tokens_details->cache_creation_tokens)) {
        $anthropicUsage->cache_creation_input_tokens = $usage->prompt_tokens_details->cache_creation_tokens;
    }

    // Add cache read tokens if present (OpenAI cached tokens)
    if (isset($usage->prompt_tokens_details->cached_tokens)) {
        $anthropicUsage->cache_read_input_tokens = $usage->prompt_tokens_details->cached_tokens;
    }
    
    return $anthropicUsage;
}

/**
 * Maps OpenAI finish_reason to Anthropic stop_reason.
 *
 * @param string $finishReason The OpenAI finish_reason value
 * @return string The corresponding Anthropic stop_reason value
 */
function openAiChatCompletionRespToAnthropicMessageResp_mapFinishReason(string $finishReason): string
{
    $mapping = [
        'stop' => 'end_turn',
        'length' => 'max_tokens',
        'tool_calls' => 'tool_use',
        'content_filter' => 'stop_sequence',
        'function_call' => 'tool_use',
    ];

    return $mapping[$finishReason] ?? 'end_turn';
}

/**
 * Checks if an OpenAI response is an error.
 *
 * @param object $response The response object to check
 * @return bool True if the response is an error
 */
function openAiChatCompletionRespToAnthropicMessageResp_isError(object $response): bool
{
    //http://192.168.112.71:8123
    if (isset($response->error) && is_object($response->error)) {
        return true;
    }
    //http://llm.ozeki.hu:10002
    if (isset($response->object) && $response->object === 'error') {
        return true;
    }
    return false;
}

/**
 * Converts an OpenAI HTTP error response to Anthropic format.
 * Useful when dealing with raw HTTP responses.
 *
 * @param object $openaiResponse The OpenAI-style error response object
 * @param int $httpCode Reference to HTTP status code variable to set
 * @return object The converted Anthropic-style error response
 */
function openAiChatCompletionRespToAnthropicMessageResp_convertError(object $openaiResponse, &$httpCode): object
{
    $error = $openaiResponse->error ?? (object)[];
    if (isset($openaiResponse->object) && $openaiResponse->object === 'error') {
        $error = $openaiResponse;
    }
    $message = $error->message ?? 'Unknown error';
    $code = $error->code ?? 500;
    $mappedError = openAiChatCompletionRespToAnthropicMessageResp_mapErrorCode($code);
    $httpCode = $mappedError['http_code'];
    return openAiChatCompletionRespToAnthropicMessageResp_createError($mappedError['type'], "OpenAI Error: " . $message);
}

/**
 * Maps OpenAI error codes to Anthropic error types and HTTP codes.
 *
 * @param int $code The OpenAI error code
 * @return array Associative array with 'type' and 'http_code'
 */
function openAiChatCompletionRespToAnthropicMessageResp_mapErrorCode(int $code) {
    $typeMapping = [
        400 => 'invalid_request_error',
        401 => 'authentication_error',
        403 => 'permission_error',
        429 => 'rate_limit_error',
        500 => 'api_error',
        503 => 'overloaded_error',
    ];

    $httpCodeMapping = [
        400 => 400,
        401 => 401,
        403 => 403,
        429 => 429,
        500 => 500,
        503 => 529,
    ];

    $type = $typeMapping[$code] ?? 'api_error';
    $httpCode = $httpCodeMapping[$code] ?? 500;
    return [
        'type' => $type,
        'http_code' => $httpCode
    ];
}

/**
 * Creates an Anthropic-style error response object.
 *
 * @param string $type  Error type (e.g. "api_error", "invalid_request")
 * @param string $message Human-readable error message
 * @return object The Anthropic-style error response object
 */
function openAiChatCompletionRespToAnthropicMessageResp_createError(string $type, string $message, string $requestId = ''): object
{
    return (object)[
        'type' => 'error',
        'error' => (object)[
            'type' => $type,
            'message' => $message,
        ],
        'request_id' => $requestId,
    ];
}
?>