<?php
/**
 * Handle request to create an API key
 *
 * @param string $userid
 * @param array  $req   Request data
 * @param array  &$resp Response data (by reference)
 * @return bool
 */

function handlerequest_createapikey($userid,$req,&$resp)
{
    $status = "CREATED";
    $error = "";

    if (!apiKeyExists($req,$error))
    {
        error_log("apikey_create.php: API key already exists. $error. Request by user $userid" . json_encode($req));
        $status = "ERROR";
    }

    else if (!saveApiKey($userid,$req,$error))
    {
        error_log("apikey_create.php: Could not save API key. $error. Request by user $userid" . json_encode($req));
        $status = "ERROR";
    }
    
    // Build response
    $resp["status"]        = "OK";
    $resp["createresult"]  = $status;
    $resp["errormessage"]  = $error;
    return true;
}

function apiKeyExists($req,&$error)
{
    global $datadir;

    $dir = "$datadir/services/aigate/apikeys/keys";
    $filename = "$dir/".md5($req["apikey"]);
    if (file_exists($filename))
    {
        $error = "Api key creation failed";
        return false;
    }

    $error = "";
    return true;
}

function saveApiKey($userid,$req,&$error)
{
    global $adminuser,$datadir;

    $tosave = [];
    $tosave["id"] = $req["id"];
    $tosave["owneruserid"] = $req["owneruserid"];
    $tosave["createdbyuserid"] = $userid;
    $tosave["comment"] = $req["comment"];
    $tosave["createdate"] = $req["createdate"];
    $tosave["signature"] = $req["signature"];

    if (!ozloadrsapublickeyof($req["owneruserid"],$publicKey,$error)) return false;
    $encryptedforowner = encryptTextRSA($publicKey,$req["apikey"]);
    $tosave["encryptedforowner"] = $encryptedforowner;

    if (!ozloadrsapublickeyof($userid,$publicKey,$error)) return false;
    $encryptedforcreatedbyuser = encryptTextRSA($publicKey,$req["apikey"]);
    $tosave["encryptedforcreatedbyuser"] = $encryptedforcreatedbyuser;

    if (!ozloadrsapublickeyof(md5($adminuser),$publicKey,$error)) return false;
    $encryptedforadmin = encryptTextRSA($publicKey,$req["apikey"]);
    $tosave["encryptedforadmin"] = $encryptedforadmin;

    $tosave["validuntil"] = mysqldateadd($tosave["createdate"],'+100 years');

    $dir = "$datadir/services/aigate/apikeys/keys";
    if (!ozmkdir($dir,$error)) return false;

    $filename = "$dir/".md5($req["apikey"]);
    if (!ozsavejson($filename,$tosave,$error)) return false;

    // Add API key ID to owner's JSON list
    if (!addApiKeyToOwnerList($req["owneruserid"], $req["id"], $error)) {
        error_log("apikey_create.php: Could not add API key ID to owner's list. $error. Request by user $userid" . json_encode($req));
    }

    $error = "";
    return true;
}

function addApiKeyToOwnerList($owneruserid, $apikeyid, &$error)
{
    global $datadir;

    $userdir = "$datadir/services/aigate/apikeys/users/$owneruserid";
    
    if (!ozmkdir($userdir, $error)) {
        return false;
    }

    $apikeysfile = "$userdir/apikeys.json";
    $apikeyslist = [];

    // Load existing API keys if the file exists
    if (file_exists($apikeysfile)) {
         if (!ozreadjson($apikeysfile, $apikeyslist, $error)) {
            $apikeyslist = [];
        }
    }

    // Add the new API key ID if it's not already in the list
    if (!in_array($apikeyid, $apikeyslist)) {
        $apikeyslist[] = $apikeyid;
    }

    // Save the updated list
    if (!ozsavejson($apikeysfile, $apikeyslist, $error)) {
        return false;
    }

    return true;
}

/*
//Input
{
id :  md5(newKey),
apikey : newKey,
action: 'createapikey',
owneruserid : userId,
comment : "",
createdate : "",
signature : "",
};

//To save
{
  "id": "9af4d8381781baccb0f915e554f8798d", //md5(apikey)
  "owneruserid": "819cbfbca5c177d076bcc5a26c5b28c2",
  "createdbyuserid": "d0eafbf473d60808db0a36d53b251b10",
  "encryptedforowner": "enc_user_data_abc123", //rsaenc(apikey,userpublickey)
  "encryptedforadmin": "enc_admin_data_xyz789",
  "encryptedforcreatedbyuser": "enc_createdby_data_mno456",
  "createdate": "2024-01-15 10:30:00",
  "validuntil": "2025-12-31 23:59:59",
  "comment": "API key for production service", 
  "signature" : "Signature of by createdby user" //rsasign(apikey+createdate,createdbyuser-publickey)
}
*/

?>