<?php
//**********************************************************
// Encryption algorithms
//**********************************************************

/**
* Define the number of blocks that should be read from the source file for each chunk.
* For 'AES-128-CBC' each block consist of 16 bytes.
* So if we read 10,000 blocks we load 160kb into memory. You may adjust this value
* to read/write shorter or longer chunks.
*/
define('FILE_ENCRYPTION_BLOCKS', 10000);

//**********************************************************
// SHA512
//**********************************************************

function generateSha512Hash($inputText) 
{
	return base64_encode(hash('sha512', $inputText));
}

function sha512($inputText)
{
    $hashedPwd = hash('sha512', $inputText);
    return strtoupper($hashedPwd);
}

//**********************************************************
// RSA
//**********************************************************

function evpkdf($passphrase, $salt) {
	$salted = '';
	$dx = '';
	while(strlen($salted) < 32) {
		$dx = md5($dx . $passphrase . $salt, true);
		$salted .= $dx;
	}
	$key = substr($salted, 0, 32);
	$iv = substr($salted, 32, 16);
	return [$key, $iv];
}

function decryptfrompkcs1($pem, $password) {	

	$pemHeader = '-----BEGIN RSA PRIVATE KEY-----';
	$pemFooter = '-----END RSA PRIVATE KEY-----';
	$pem = str_replace($pemHeader, '', $pem);
	$pem = str_replace($pemFooter, '', $pem);

	$pemList = preg_split('/\r\n|\n|\r/', $pem, -1, PREG_SPLIT_NO_EMPTY);
	
	$dekInfo = explode(',', explode(': ', $pemList[1])[1]);
	
	$alg = trim($dekInfo[0]);
	$iv = trim($dekInfo[1]);
	$ivl = openssl_cipher_iv_length($alg);
	$salt = substr($iv, 0, $ivl);

	$keyparts = array_slice($pemList, 2);
	$encryptedPem = implode($keyparts);
	
	list($key) = evpkdf($password, hex2bin($salt));
	
	$decrypted = openssl_decrypt($encryptedPem, $alg, $key, OPENSSL_ZERO_PADDING, hex2bin($iv));
	$decryptedBase64 = base64_encode($decrypted);
	return $pemHeader . "\r\n" . $decryptedBase64 . "\r\n" . $pemFooter;
}

function encrypttopkcs1($pem, $password) {
	$pemHeader = '-----BEGIN RSA PRIVATE KEY-----';
	$pemFooter = '-----END RSA PRIVATE KEY-----';
	$pem = str_replace($pemHeader, '', $pem);
	$pem = str_replace($pemFooter, '', $pem);
	
	$pemHeader2 = '-----BEGIN PRIVATE KEY-----';
	$pemFooter2 = '-----END PRIVATE KEY-----';
	$pem = str_replace($pemHeader2, '', $pem);
	$pem = str_replace($pemFooter2, '', $pem);
	
	//$alg = 'AES-128-CFB';
	$alg = 'AES-256-CFB';
	$ivl = openssl_cipher_iv_length($alg);
	$iv = bin2hex(openssl_random_pseudo_bytes($ivl));
	$salt = substr($iv, 0, $ivl);
	
	list($key) = evpkdf($password, hex2bin($salt));
	
	$decryptedPem = trim(preg_replace('/\r|\n/', '', $pem));
	$decryptedPemBytes =  base64_decode($decryptedPem);
	
	$encrypted = openssl_encrypt($decryptedPemBytes, $alg, $key, OPENSSL_RAW_DATA | OPENSSL_ZERO_PADDING, hex2bin($iv));
	$encryptedBase64 = base64_encode($encrypted);
	$encryptedBase64 = chunk_split($encryptedBase64, 64, "\r\n");
	
	return $pemHeader . "\r\nProc-Type: 4,ENCRYPTED\r\nDEK-Info: " . $alg . "," . $iv . "\r\n\r\n" . trim($encryptedBase64) . "\r\n" . $pemFooter;
}


function getOpenSSLConfig()
{
	$config = [];
	global $opensslconf;
	if ($opensslconf!="") $config["config"]=$opensslconf;
	if ($opensslconf=="" && isWindows()) $config["config"]= getOpenSSLConfigOnWindows();
	return $config;
}

function generateRsaKeypair(&$privateKey, &$publicKey,&$error) : bool
{
	$config = array(
		"digest_alg" => "RSA-SHA512",
		"private_key_bits" => 2048,
		"private_key_type" => OPENSSL_KEYTYPE_RSA,
		);

	$config = getOpenSSLConfig();
	$privateKey = openssl_pkey_new($config);
	if ($privateKey == null)
	{
		$error = "OpenSSL key generation error. OpenSSL is probably not installed or it is not configured properly.";
		return false;
	}
	
	$keyDetails = openssl_pkey_get_details($privateKey);
	$publicKeyPem = $keyDetails["key"];
	$publicKey = openssl_pkey_get_public($publicKeyPem);
	return true;
}

function exportRsaPrivateKeyToPEM($privateKey, $password=null)
{
	if (!exportRsaPrivateKeyAsPEM($privateKey, $password, $pemPkcs1Encrypted,$error)) return "";
	return $pemPkcs1Encrypted;
}

function exportRsaPrivateKeyAsPEM($privateKey, $password, &$pemPkcs1Encrypted,&$error)
{
	//pkcs1 formátumban kell exportálni
	//olyan formátumban, hogy C#-ban tudjuk olvasni
	$privateKeyRes = openssl_pkey_get_private($privateKey);
	if (!is_resource($privateKeyRes) && !is_object($privateKeyRes)) {
		$error = "The private key is not a resources";
		return false;
	}

	$config = getOpenSSLConfig();
	openssl_pkey_export($privateKeyRes, $pemPkcs8, null, $config);
	if (!isset($pemPkcs8) || $pemPkcs8==NULL) 
	{
		$error = "Open SSL key export failed.";
		return false;
	}

	$pemPkcs1 = convertPkcs8ToPkcs1($pemPkcs8);
	if ($password==null) 
	{
		$pemPkcs1Encrypted = $pemPkcs1;
		$error = "";
		return true;
	}

	$pemPkcs1Encrypted = encrypttopkcs1($pemPkcs1, $password);
	$error = "";
	return true;
}

function exportRsaPublicKeyToPEM($key)
{
	$keyDetails = openssl_pkey_get_details($key);
	
	$publicKeyPem = $keyDetails["key"];
	$publicKey = openssl_pkey_get_public($publicKeyPem);
	$publicKeyRes = openssl_pkey_get_public($publicKey);
	$keyDetails = openssl_pkey_get_details($publicKeyRes);
	$pem = $keyDetails["key"];
	return $pem;
}

function formatPrivateKey($PrivateKey)
{	
	$pemHeader = '-----BEGIN RSA PRIVATE KEY-----';
	$pemFooter = '-----END RSA PRIVATE KEY-----';
	$PrivateKey = str_replace($pemHeader, '', $PrivateKey);
	$PrivateKey = str_replace($pemFooter, '', $PrivateKey);
	
	$pemList = preg_split('/\r\n|\n|\r/', $PrivateKey, -1, PREG_SPLIT_NO_EMPTY);	
	$keyparts = array_slice($pemList, 2);
	$encryptedPem = implode($keyparts);
	
	return $pemHeader . "\r\n". $pemList[0]. "\r\n" . $pemList[1] . "\r\n". "\r\n". $encryptedPem . "\r\n" . $pemFooter;
}

function importRsaPrivateKeyFromPEM($privateKeyPem, $password, &$privateKey, &$publicKey, &$error)
{
	$privateKeyPem = trim($privateKeyPem);

	if ($password==null) {
		$decrypted = $privateKeyPem;
	} else {
		$decrypted = decryptfrompkcs1($privateKeyPem, $password);
	}
	$privateKey = openssl_pkey_get_private($decrypted);
	if (!is_resource($privateKey) && !is_object($privateKey)) 
	{
		$error = "Could not import RSA private key.";
		return false;
	}

	$keyDetails = openssl_pkey_get_details($privateKey);
	$publicKeyPem = $keyDetails["key"];
	$publicKey = openssl_pkey_get_public($publicKeyPem);

	return true;
}

function isPrivateKeyPem($privateKeyPem, &$error)
{
	return mb_strpos(strtolower($privateKeyPem), "private key") !== false;
}

function importRsaPublicKeyFromPEM($publicKeyPem, &$publicKey,&$error)
{
	$publicKeyPem = trim($publicKeyPem);
	$publicKey = openssl_pkey_get_public($publicKeyPem);
	if (!is_resource($publicKey) && !is_object($publicKey))
	{
		$error = "Could not import public key";
		return false;
	}
	$error = "";
	return true;
}

function isPublicKeyPem($publicKeyPem, &$error)
{
	return !importRsaPublicKeyFromPEM($publicKeyPem, $publicKey,$error);
}

function formatPublicKey($publicKey)
{	
	if (is_resource($publicKey)) return $publicKey;
	if (is_object($publicKey)) return $publicKey;
	$pemHeader = '-----BEGIN PUBLIC KEY-----';
	$pemFooter = '-----END PUBLIC KEY-----';
	$publicKeyRaw = $publicKey;
	$publicKeyRaw = str_replace($pemHeader, '', $publicKeyRaw);
	$publicKeyRaw = str_replace($pemFooter, '', $publicKeyRaw);
	$publicKeyRaw = preg_replace('/\r\n|\r|\n/', '', $publicKeyRaw);
	return $pemHeader . "\r\n" . $publicKeyRaw . "\r\n" . $pemFooter;
}

function convertPkcs8ToPkcs1($pemPkcs8)
{
	$pemHeader = '-----BEGIN PRIVATE KEY-----';
	$pemFooter = '-----END PRIVATE KEY-----';
	
	$dataPkcs8 = $pemPkcs8;
	$dataPkcs8 = str_replace($pemHeader, '', $dataPkcs8);
	$dataPkcs8 = str_replace($pemFooter, '', $dataPkcs8);
	$dataPkcs8 = preg_replace('/\r\n|\r|\n/', '', $dataPkcs8);
	
	$dataDer = base64_decode($dataPkcs8);
	$dataDer = unpack('H*', $dataDer)[1];
	
	$dataPkcs1 = substr($dataDer, 38 + 6 + 2 + 6);
	$dataPkcs1 = pack('H*', $dataPkcs1);
	$dataPkcs1 = base64_encode($dataPkcs1);
	$pemPkcs1 =  "-----BEGIN RSA PRIVATE KEY-----\r\n" . chunk_split($dataPkcs1, 64, "\r\n") . '-----END RSA PRIVATE KEY-----';
	return $pemPkcs1;
}

function encryptTextRSA($publicKey,$plainText,$padding=OPENSSL_PKCS1_PADDING)
{	
	if (!is_resource($publicKey) && (!is_object($publicKey))) $publicKey = formatPublicKey($publicKey);
	openssl_public_encrypt($plainText,$encrpytedText,$publicKey,$padding);
  	return base64_encode($encrpytedText);
}

function decryptTextRSA($privateKey,$encrypted,&$decrypted,&$error)
{
	if (!is_resource($privateKey) && !is_object($privateKey))
	{
		$error = "Invalid private key provided";
		return false;		
	}

	if (isEmpty($encrypted))
	{
		$error = "Cannot decrypt empty text";
		return false;		
	}

	$enc = base64_decode($encrypted);
	
	if (!openssl_private_decrypt($enc,$plainText,$privateKey))
	{
		$errormessage = error_get_last();
		if (is_array($errormessage)) $errormessage=implode("<BR>\r\n",$errormessage);
		$error = "RSA decryption failed. ".$errormessage;
		return false;
	}

	if (isEmpty($plainText))
	{
		$error = "RSA decryption return empty text";
		return false;
	}

	$decrypted = $plainText;
	return true;
}

function signRSA($privateKey,$text,&$signature,&$error,$algo=OPENSSL_ALGO_SHA256)
{
	if (!is_resource($privateKey) && !is_object($privateKey))
	{
		$error = "The private key is invalid.";
		return false;
	}
	if (isEmpty($text))
	{
		$error = "Cannot sign an empty text";
		return false;
	}

	if (!openssl_sign($text, $signature, $privateKey, $algo))
	{
		$error = "The signature failed";
		return false;
	}

	$signature = base64_encode($signature);
	return true;
}

function isSignatureValid($text,$signature,$publicKey,&$error,$algo=OPENSSL_ALGO_SHA256)
{
	global $debug;

	if (isEmpty($signature))
	{
		$error = "The signature cannot be empty";
		return false;
	}

	if (!is_resource($publicKey) && !is_object($publicKey))
	{
		$error = "The public key must be provided";
		return false;
	}

	$sign = base64_decode($signature);

	$ok = openssl_verify($text, $sign, $publicKey, $algo);
	
	if (!$ok) {
		$error = "The signature verification failed.";
		if ($debug)
		{
			$error .= "Text: ".strlen($text)." bytes. Signature: ".strlen($signature)." bytes. ".
			"Public key: ".(($publicKey instanceof OpenSSLAsymmetricKey)?"present":"not present")."; Algo: ".$algo."; ";
			if ($ok === 0) {
				$error .= "Signature is invalid.";
			} else {
				$error .= "Error verifying signature: " . openssl_error_string();
			}
		}
    	return false;
	}

   	return true;
}

//**********************************************************
// AES 256
//**********************************************************

function generateAesKey()
{
	$aeskey = openssl_random_pseudo_bytes(32);
	return base64_encode($aeskey);
}

function convertToAesKey($password)
{
	$aeskey = md5($password,true);
	$aeskey .= md5($aeskey,true); //AES 256 
	return base64_encode($aeskey);
}

function encryptTextAES($aesKey, $plainText)
{
	return encryptTextAES256($aesKey, $plainText);
}

function decryptTextAES($aesKey, $encrpytedText)
{
	$ret = decryptTextAES256($aesKey, $encrpytedText);
	if ($ret == null)
	{
		$ret = decryptTextAES128($aesKey, $encrpytedText);
	}
	return $ret;
}

function encryptTextAES128($aesKey, $plainText)
{
	$method = "AES-128-CBC";
	$key = base64_decode($aesKey);	
	$ivl = openssl_cipher_iv_length($method);
	$iv = openssl_random_pseudo_bytes($ivl);
	$ciphertext = openssl_encrypt($plainText, $method, $key, OPENSSL_RAW_DATA, $iv);		
    $ret = base64_encode($iv . $ciphertext);     
	return $ret;
}

function decryptTextAES128($aesKey, $encrpytedText)
{
	$method = "AES-128-CBC";
	$key = base64_decode($aesKey);	
	$decoded = base64_decode($encrpytedText);
	$ivl = openssl_cipher_iv_length($method);
	$iv = substr($decoded, 0, $ivl);	
	$ciphertext = substr($decoded, $ivl);	
	$decrypted = openssl_decrypt($ciphertext, $method, $key, OPENSSL_RAW_DATA, $iv);
	return $decrypted;
}

function encryptTextAES256($aesKey, $plainText)
{
	$method = "AES-256-CBC";
	$key = base64_decode($aesKey);	
	$ivl = openssl_cipher_iv_length($method);
	$iv = openssl_random_pseudo_bytes($ivl);
	$ciphertext = openssl_encrypt($plainText, $method, $key, OPENSSL_RAW_DATA, $iv);		
    $ret = base64_encode($iv . $ciphertext);     
	return $ret;
}

function decryptTextAES256($aesKey, $encrpytedText)
{
	$method = "AES-256-CBC";
	$key = base64_decode($aesKey);	
	$decoded = base64_decode($encrpytedText);
	$ivl = openssl_cipher_iv_length($method);
	if ($ivl<16) return "";
	$iv = substr($decoded, 0, $ivl);	
	$ciphertext = substr($decoded, $ivl);	
	//cipher expects an IV of precisely 16 bytes, padding with \0
	$decrypted = openssl_decrypt($ciphertext, $method, $key, OPENSSL_RAW_DATA, $iv);
	return $decrypted;
}


//**********************************************************
// AES 256 files
//**********************************************************

function encryptFileAES($aesKey, $source, $dest, &$error)
{
	$method = 'AES-256-CBC';
	$key = base64_decode($aesKey);
	$iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));
	
	if(!file_exists($source)) 
	{
		$error = "File to encrypt does not exist";
		return false;
	}
	$fpSource = fopen($source, 'rb');
	if(!$fpSource) 
	{
		$error = "Cannot open source file to encrypt";
		return false;
	}
	
	$fpDest = fopen($dest, 'w');
	if(!$fpDest) {
		fclose($fpSource);
		$error = "Cannot open destination file to encrypt";
		return false;
	}
	
	// Put the initialzation vector to the beginning of the file
	fwrite($fpDest, $iv);
	while (!feof($fpSource)) {
		$plaintext = fread($fpSource, 16 * FILE_ENCRYPTION_BLOCKS);
		$ciphertext = openssl_encrypt($plaintext, $method, $key, OPENSSL_RAW_DATA, $iv);
		if($ciphertext === false) {
			fclose($fpSource);
			fclose($fpDest);
			$error = "Error while creating cypher data";
			return false;
		}
		// Use the first 16 bytes of the ciphertext as the next initialization vector
		$iv = substr($ciphertext, 0, 16);
		fwrite($fpDest, $ciphertext);
	}
	
	fclose($fpSource);
	fclose($fpDest);
	return true;
}

function decryptFileAES($aesKey, $source, $dest,&$error)
{
	$method = 'AES-256-CBC';
	$key = base64_decode($aesKey);
	
	if(!file_exists($source)) {
		$error = "Source file does not exist";
		return false;
	}

	$fpSource = fopen($source, 'rb');
	if(!$fpSource) 
	{
		$error = "Could not open file";
		return false;
	}
	
	$fpDest = fopen($dest, 'w');
	if(!$fpDest) {
		fclose($fpSource);
		$error = "Could not open destination file";
		return false;
	}
	
	// Get the initialzation vector from the beginning of the file
	$iv = fread($fpSource, 16);
	while (!feof($fpSource)) {
		// We have to read one block more for decrypting than for encrypting
		$ciphertext = fread($fpSource, 16 * (FILE_ENCRYPTION_BLOCKS + 1));
		$plaintext = openssl_decrypt($ciphertext, $method, $key, OPENSSL_RAW_DATA, $iv);
		if($plaintext === false) {
			fclose($fpSource);
			fclose($fpDest);
			$error = "Could not decrypt source file";
			return false;
		}
		// Use the first 16 bytes of the ciphertext as the next initialization vector
		$iv = substr($ciphertext, 0, 16);
		fwrite($fpDest, $plaintext);
	}
	
	fclose($fpSource);
	fclose($fpDest);
	return true;
}

function loadFileAES($aesKey, $source, &$content, &$error)
{
	$method = 'AES-256-CBC';
	$key = base64_decode($aesKey);
	
	if(!file_exists($source)) {
		$error = "Source file does not exist";
		return false;
	}

	$fpSource = fopen($source, 'rb');
	if(!$fpSource) 
	{
		$error = "Could not open file";
		return false;
	}
	
	$contentpart = array();
	// Get the initialzation vector from the beginning of the file
	$iv = fread($fpSource, 16);
	while (!feof($fpSource)) {
		// We have to read one block more for decrypting than for encrypting
		$ciphertext = fread($fpSource, 16 * (FILE_ENCRYPTION_BLOCKS + 1));
		$plaintext = openssl_decrypt($ciphertext, $method, $key, OPENSSL_RAW_DATA, $iv);
		if($plaintext === false) {
			fclose($fpSource);
			$error = "Could not decrypt source file";
			return false;
		}
		// Use the first 16 bytes of the ciphertext as the next initialization vector
		$iv = substr($ciphertext, 0, 16);
		$contentpart[] = $plaintext;
	}
	
	fclose($fpSource);
	
	$content = implode("",$contentpart);

	return true;
}

function saveFileAES($aesKey, $dest, $content, &$error)
{
	$method = 'AES-256-CBC';
	$key = base64_decode($aesKey);
	$iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));
	
	$fpDest = fopen($dest, 'w');
	if(!$fpDest) {
		$error = "Cannot open destination file to encrypt";
		return false;
	}
	
	// Put the initialzation vector to the beginning of the file
	fwrite($fpDest, $iv);

	$plaintext = $content;
	$ciphertext = openssl_encrypt($plaintext, $method, $key, OPENSSL_RAW_DATA, $iv);
	if($ciphertext === false) {
		fclose($fpDest);
		$error = "Error while creating cypher data";
		return false;
	}
	// Use the first 16 bytes of the ciphertext as the next initialization vector
	$iv = substr($ciphertext, 0, 16);
	fwrite($fpDest, $ciphertext);
	
	fclose($fpDest);
	return true;
}

//**********************************************************
// RSA AES
// -------
// AES key: 16 bytes 
// RSA alg: 'AES-256-CBC'
// -------
// $paddingbytes [4 bytes]: longint
// $keybytelength [4 bytes]: longing
// $encrcyptedaeskey [256 bytes]:
// $encryptedtext
//*****************************************************

function encryptTextRSAAES($publicKey, $plainText,$method = 'AES-256-CBC')
{
	if (!is_resource($publicKey) && (!is_object($publicKey))) $publicKey = formatPublicKey($publicKey);

	$aeskey = openssl_random_pseudo_bytes(16);//FIXME
	//$aeskey = openssl_random_pseudo_bytes(openssl_cipher_key_length($method));
    $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));

    $paddingLength = strlen($aeskey) - strlen($plainText) % strlen($aeskey);
    $paddingbytes = pack("L", $paddingLength); //Unsigned long: 32 bit

    $encryptedText = openssl_encrypt($plainText, $method, $aeskey, OPENSSL_RAW_DATA,$iv);
    $encryptedText = $iv.$encryptedText;
    
    $ok = openssl_public_encrypt($aeskey, $encrcyptedaeskey, $publicKey,OPENSSL_PKCS1_OAEP_PADDING);

    if ($ok != true){
        while ($msg = openssl_error_string())
	    {
            echo $msg . "<br/>\n";
        }
    }

    $keybytelength = pack("L", strlen($encrcyptedaeskey));
    $encryptedText = base64_encode($paddingbytes.$keybytelength.$encrcyptedaeskey.$encryptedText);
    return $encryptedText;
}

function decryptTextRSAAES($privateKey, $encrpytedText, &$decrypted ,&$error)
{
	if ($privateKey==NULL)
	{
		$error = "Cannot decrypt RSA data. Invalid private key";
		return false;
	}

	$encrpytedText = base64_decode($encrpytedText);
	if ($encrpytedText=="")
	{
		$error = "Cannot decrypt empty text. RSA AES decrpyt failed.";
		return false;
	}

    $paddingbytes = substr($encrpytedText, 0, 4);
    $paddinglength = unpack("I",$paddingbytes);

    $keybytes = substr($encrpytedText, 4, 4);
    $keylength = unpack("I",$keybytes);

    $encrcyptedaeskey = substr($encrpytedText, 8, $keylength[1]);

    $encryptedBytes = substr($encrpytedText, 8+$keylength[1], (strlen($encrpytedText)-8-$keylength[1]));

    $ok = openssl_private_decrypt($encrcyptedaeskey, $decryptedaeskey, $privateKey, OPENSSL_PKCS1_OAEP_PADDING);

    if ($ok != true){
        while ($msg = openssl_error_string())
	    {
            $error.=$msg . "<br/>\n";
        }
		return false;
    }

	$aesalg = 'AES-256-CBC';
	$ivl = openssl_cipher_iv_length($aesalg);
    $iv = substr($encryptedBytes, 0, $ivl);
    $encryptedBytesWithoutIV = substr($encryptedBytes, strlen($iv));

	$decrypted = openssl_decrypt($encryptedBytesWithoutIV, $aesalg, $decryptedaeskey,OPENSSL_RAW_DATA,$iv);
	
	if ($decrypted=="")
	{
		$aesalg = 'AES-128-CBC';
		$ivl = openssl_cipher_iv_length($aesalg);
		$iv = substr($encryptedBytes, 0, strlen($decryptedaeskey));
		$encryptedBytesWithoutIV = substr($encryptedBytes, strlen($iv));
		$decrypted = openssl_decrypt($encryptedBytesWithoutIV, $aesalg, $decryptedaeskey,OPENSSL_RAW_DATA,$iv);
	}

	if (isempty($decrypted) || $decrypted===false)
	{
		$error = "Decryption returned 0 bytes<br>";
		$error.= "Padding length bytes: 4<BR>";
		$error.= "Padding length value: ".$paddinglength[1]."<BR>";
		$error.= "AES key length bytes: 4<BR>";
		$error.= "AES key length value: ".$keylength[1]."<BR>";
		$error.= "AES key bytes: 32<BR>";
		$error.= "AES alg: $aesalg<BR>";
		$error.= "AES key length: ".strlen($decryptedaeskey)."<BR>";
		$error.= "AES key decoded: ".base64_encode($decryptedaeskey)."<BR>";
		$error.= "RSA Algorithm: AES-256-CBC.<BR>";
		$error.= "IV + Encrpyted data length: ".strlen($encryptedBytes)."<BR>";
		$error.= "IV + Encrpyted data: ".base64_encode($encryptedBytes)."<BR>";
		$error.= "IV bytes: ".strlen($decryptedaeskey)."<BR>";
		$error.= "IV length: ".$ivl."<BR>";
		$error.= "IV: ".base64_encode($iv)."<BR>";
		$error.= "Data bytes: ".strlen($encryptedBytesWithoutIV)."<BR>";
		$error.= "EncryptedData: ".base64_encode($encryptedBytesWithoutIV)."<BR>";
		return false;
	}

    return true;
}

function encryptFileRSAAES($publicKey, $source, $dest, &$error)
{
	$method = 'AES-256-CBC';
	$aesKey = openssl_random_pseudo_bytes(16);
	$iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));
	
	if(!file_exists($source)) 
	{
		$error = "Source file does not exist";
		return false;
	}
	$sourceSize = filesize($source);
	if(!$sourceSize) 
	{
		$error = "Source file is empty";
		return false;
	}
	$paddingLength = strlen($aesKey) - $sourceSize % strlen($aesKey);
	$paddingBytes = pack('L', $paddingLength);
	
	if(!openssl_public_encrypt($aesKey, $aesKeyEncrypted, $publicKey, OPENSSL_PKCS1_OAEP_PADDING)) 
	{
		$error = "AES encryption error";
		return false;
	}
	$keyBytes = pack('L', strlen($aesKeyEncrypted));
	
	$fpSource = fopen($source, 'rb');
	if(!$fpSource) 
	{
		$error = "Cannot open source file for encryption";
		return false;
	}
	
	$fpDest = fopen($dest, 'w');
	if(!$fpDest) {
		fclose($fpSource);
		$error = "Cannot open destination file for encryption";
		return false;
	}
	stream_filter_append($fpDest, 'convert.base64-encode');
	
	fwrite($fpDest, $paddingBytes);
	fwrite($fpDest, $keyBytes);
	fwrite($fpDest, $aesKeyEncrypted);
	fwrite($fpDest, $iv);
	while(!feof($fpSource)) {
		$plaintext = fread($fpSource, 16 * FILE_ENCRYPTION_BLOCKS);
		$ciphertext = openssl_encrypt($plaintext, $method, $aesKey, OPENSSL_RAW_DATA, $iv);
		if($ciphertext === false) {
			fclose($fpSource);
			fclose($fpDest);
			return false;
		}
		// Use the first 16 bytes of the ciphertext as the next initialization vector
		$iv = substr($ciphertext, 0, 16);
		fwrite($fpDest, $ciphertext);
	}
	
	fclose($fpSource);
	fclose($fpDest);
	return true;
}

function decryptFileRSAAES($privateKey, $source, $dest, &$error)
{
	$method = 'AES-256-CBC';

	if (!is_resource($privateKey) && !is_object($privateKey))
	{
		$error = "The provided priate key is not an RSA key resource";
		return false;
	}
	
	if(!file_exists($source)) {
		$error = "Source file does not exist";
		return false;
	}
	$sourceSize = filesize($source);
	if(!$sourceSize) 
	{
		$error = "Source file has 0 length";
		return false;
	}
	
	$fpSource = fopen($source, 'rb');
	if(!$fpSource) 
	{
		$error = "Could not open source file";
		return false;
	}
	stream_filter_append($fpSource, 'convert.base64-decode');
	
	$paddingBytes = fread($fpSource, 4);
	$paddingLength = unpack('I', $paddingBytes);
	$keyBytes = fread($fpSource, 4);
	$keyLength = unpack('I', $keyBytes);
	
	$aesKeyEncrypted = fread($fpSource, $keyLength[1]);
	if(!openssl_private_decrypt($aesKeyEncrypted, $aesKeyDecrypted, $privateKey, OPENSSL_PKCS1_OAEP_PADDING)) {
		fclose($fpSource);
		$error = "RSA decrypt failed";
		return false;
	}
	
	$fpDest = fopen($dest, 'w');
	if(!$fpDest) {
		fclose($fpSource);
		$error = "Cannot open destination file";
		return false;
	}
	
	$iv = fread($fpSource, strlen($aesKeyDecrypted));
	while (!feof($fpSource)) {
		// We have to read one block more for decrypting than for encrypting
		$ciphertext = fread($fpSource, 16 * (FILE_ENCRYPTION_BLOCKS + 1));
		$plaintext = openssl_decrypt($ciphertext, $method, $aesKeyDecrypted, OPENSSL_RAW_DATA, $iv);
		if($plainText === false) {
			fclose($fpSource);
			fclose($fpDest);
			$error = "AES decrypt failed";
			return false;
		}
		// Use the first 16 bytes of the ciphertext as the next initialization vector
		$iv = substr($ciphertext, 0, 16);
		fwrite($fpDest, $plaintext);
	}
	
	fclose($fpSource);
	fclose($fpDest);
	return true;
}

//**********************************************************
// Encrypted file storage
//**********************************************************

function ozsaveencrypted($aeskey,$filename,$content,&$error)
{
	if (strlen($aeskey)==0)
	{
		$error = "No AES key loaded";
		return false;
	}
	$encrypted = encryptTextAES($aeskey,$content);
	if (!file_put_contents($filename, $encrypted))
	{
		$e = error_get_last();
	 	$errormessage = (isset($e) && isset($e['message']) && $e['message'] != "") ? $e['message'] : "";
		$error = "Cannot save file $filename. ".$errormessage;
		return false;
	}
	$error = "";
	return true;
}

function ozloadencrypted($aeskey,$filename,&$content,&$error)
{
	if (strlen($aeskey)==0)
	{
		$error = "No AES key loaded";
		$content = "";
		return false;
	}
	$text = @file_get_contents($filename);
	if ($text===false)
	{
		$e = error_get_last();
	 	$errormessage = (isset($e) && isset($e['message']) && $e['message'] != "") ? $e['message'] : "";
		$error = "Cannot load file $filename. ".$errormessage;
		$content = "";
		return false;
	}
	$content = decryptTextAES($aeskey,$text);
	$error = "";
	return true;
}

function ozloadencryptedjson($aeskey,$filename,&$arr,&$error)
{
	if (!file_exists($filename))
	{
		$error = "File does not exist";
		return false;
	}

	if (isEmpty($aeskey))
	{
		if (!ozreadjson($filename,$arr,$error))
		{
			$error = "Could not load content. $error";
			return false;
		}
		return true;
	}

	if (!ozloadencrypted($aeskey,$filename,$content,$error))
	{
		$error = "Could not load encrypted content. $error";
		return false;
	}

	if (isEmpty($content))
	{
		$error = "Could not load encrypted content. Empty content found";
		return false;
	}

	$arr = json_decode($content,true);
	if ($arr===false)
	{
		$error = "Could not decode json data. ".json_last_error();
		return false;
	}

	return true;
}

function ozsaveencryptedjson($aeskey,$filename,$arr,&$error)
{
	if (isEmpty($aeskey))
	{
		$error = "Encrypted json save failed. Invalid AES key.";
		return false;
	}

	if (!is_array($arr))
	{
		$error = "Encrypted json save failed. Cannot json encode an empty array.";
		return false;
	}

	$content = json_encode($arr);
	if (isEmpty($content))
	{
		$error = "Encrypted json save failed. Json encoding failed.";
		return false;
	}

	if (!ozsaveencrypted($aeskey,$filename,$content,$errorm))
	{
		$error = "Encrypted json save failed. Could not save file. $errorm";
		return false;
	}

	return true;
}

function ozloadencryptedrsa($privatekey,$filename,&$content,&$error)
{
	if (!is_resource($privatekey) && !is_object($privatekey))
	{
		$error = "File decrytption failed. No RSA private key loaded";
		$content = "";
		return false;
	}

	if (!file_exists($filename))
	{
		$error = "File decrytption failed. File to decrypt does not exist.";
		return false;
	}

	$text = @file_get_contents($filename);
	if ($text===false)
	{
		$e = error_get_last();
	 	$errormessage = (isset($e) && isset($e['message']) && $e['message'] != "") ? $e['message'] : "";
		$error = "File decrytption failed. Cannot load file $filename. ".$errormessage;
		$content = "";
		return false;
	}

	if (!decryptTextRSA($privatekey,$text,$content,$errorm))
	{
		$error = "File decrytption failed. RSA decryption error. ".$errorm;
		return false;
	}

	$error = "";
	return true;
}

function ozloadencryptedrsaaes($privatekey,$filename,&$content,&$error)
{
	if (!is_resource($privatekey) && !is_object($privatekey))
	{
		$error = "File decrytption failed. No RSA private key loaded";
		$content = "";
		return false;
	}

	if (!file_exists($filename))
	{
		$error = "File decrytption failed. File to decrypt does not exist.";
		return false;
	}

	$text = @file_get_contents($filename);
	if ($text===false)
	{
		$e = error_get_last();
	 	$errormessage = (isset($e) && isset($e['message']) && $e['message'] != "") ? $e['message'] : "";
		$error = "File decrytption failed. Cannot load file $filename. ".$errormessage;
		$content = "";
		return false;
	}

	if (!decryptTextRSAAES($privatekey,$text,$content,$errorm))
	{
		$error = "File decrytption failed. RSA decryption error. ".$errorm;
		return false;
	}

	$error = "";
	return true;
}

//**********************************************************
// Setup system aes key
//**********************************************************
if (!isset($siteid) || isEmpty($siteid))
{
    $siteid = generateAesKey();
	$configfile = $configdir."/config.php";
	$config = file_get_contents($configfile);
	if ($config != false && !(str_contains($config,'$siteid')))
	{
		$siteidtxt = "\$siteid = \"$siteid\";\r\n\$siteurl";
		$config = str_replace('$siteurl',$siteidtxt,$config);
		file_put_contents($configfile,$config);
	}
}

//**********************************************************
// PHP <8.2.0
//**********************************************************
if(!function_exists('openssl_cipher_key_length'))
{
	function openssl_cipher_key_length($cipher_algo)
	{
		$cipher_algo=strtolower($cipher_algo);
		if(str_starts_with($cipher_algo,"aes-128"))return 16;
		if(str_starts_with($cipher_algo,"aes-192"))return 24;
		if(str_starts_with($cipher_algo,"aes-256"))return 32;
		return false;
	}
}

?>
