<?php

//******************************************************
// Input validation
//******************************************************
function ozisdir($dir)
{
	if (isEmpty($dir)) return false;
	return is_dir($dir);
}

function issafefile($filename)
{
	return isfilesafe($filename,$error);
}

function isfilesafe($filename,&$error)
{
	if (!ozfilenamesafe($filename,$safefilename,$error,true))
	{
		return false;
	}
	if ($filename != $safefilename)
	{
		$error = "This is not a safe file name: $filename";
		return false;
	}
	return true;
}

function ozfilenamesafe($filename, &$safefilename, &$error, $space_to_underscore = false)
{
	//remove @ characters
	$safefilename = preg_replace('/[@\.:]/u', '_', $filename);

	//remove unicode characters
	$safefilename = preg_replace('/[\x00-\x1F\x7F]/u', '', $safefilename);
	
	if (!$space_to_underscore) {
		// Remove anything which isn't a word, number or any of the following caracters -~,;[]().
		$safefilename = preg_replace("([^\w\d\-~,;\[\]\(\).])", '', $safefilename);
	} else {
		// Remove anything which isn't a word, number or any of the following caracters -~,;[]().
		$safefilename = preg_replace("([^\w\d\-~,; \[\]\(\).])", '', $safefilename);

		// Replace multiple spaces with single underscores
		$safefilename = preg_replace('/\s+/', '_', $safefilename);
	}
	
	// Remove any runs of periods 
	$safefilename = preg_replace("([\.]{2,})", '', $safefilename);

	if (empty($safefilename))
	{
		$safefilename = "";
		$error = "Empty file name received.";
		return false;
	}

	$error = "";
	return true;
}

function oziptofilename($ip)
{
	$ret = mb_ereg_replace("([^\w\s\d\-_~,;\[\]\(\).])", '', $ip);
	$ret = mb_ereg_replace("([\.]{2,})", '', $ret);
	return $ret;
}

//**********************************************************
// Json file management
//**********************************************************
function ozreadjson($f,&$jsondecoded,&$error)
{
    if (!file_exists($f)) 
	{
		$error = "File does not exists;";
		$content = "";
		return false;
	}

	if (!isInValidDataDir($f))
	{
		$error = "File json read is not safe outside the data dir.";
		return false;
	}

	if (($content = @file_get_contents($f)) === false) 
	{
		$error = error_get_last();
	    return false;
  	}

	$jsondecoded = json_decode($content,true);
	if ($jsondecoded==null)
	{
		$error = "JSON data empty. $f";
		return false;
	}
	return true;
}

function ozsavejson($f,$jsonobj,&$error,$flags=null)
{
	if (!isInValidDataDir($f))
	{
		$error = "File json write is not safe outside the data dir.";
		return false;
	}

	if ($flags!=null)
	{
		$json = json_encode($jsonobj,$flags);	
	} else {
		$json = json_encode($jsonobj);
	}
	return ozsavefile($f, $json, $error);
}

//**********************************************************
// Txt file management
//**********************************************************
function ozreadfile($f,&$content,&$error,$validatedir=true)
{
    if (!file_exists($f)) 
	{
		$error = "File does not exists;";
		$content = "";
		return false;
	}

	if ($validatedir&&!isInValidDataDir($f))
	{
		$error = "File read is not safe outside the data dir.";
		return false;
	}

	if (($content = @file_get_contents($f)) === false) 
	{
		$error = error_get_last()["message"];
	    return false;
  	}
	return true;
}

function ozsavefile($path, $content, &$error)
{
	if (!isInValidDataDir($path))
	{
		$error = "File save is not safe outside the data dir.";
		return false;
	}

	if (@file_put_contents($path,$content)===false)
	{
		$error = "Could not save file to ".$path."; ".error_get_last()["message"];
		return false;
	}

	return true;
}

function ozappendfile($path, $content, &$error)
{
	if (!isInValidDataDir($path))
	{
		$error = "File append is not safe outside the data dir.";
		return false;
	}

	if (!file_put_contents($path,$content,FILE_APPEND))
	{
		$error = "Could not save file to ".$path."; ".error_get_last()["message"];
		return false;
	}

	return true;
}

//******************************************************
// Txt files containing lists
//******************************************************
function ozreadlines($f,&$lines,&$error,$validatedir=true)
{
    if (!file_exists($f)) 
	{
		$error = "File does not exist";
		$lines = array();
		return true;
	}

	if ($validatedir&&!isInValidDataDir($f))
	{
		$error = "File read is not safe outside the data dir.";
		$lines = array();
		return false;
	}

	if (!ozreadfile($f,$content,$error,$validatedir))
	{
		$lines = array();
		return false;
	}
	if ($content == "")
	{
		$lines = array();
		return true;
	}
	$lines = preg_split("/\r\n|\n|\r/", $content,-1,PREG_SPLIT_NO_EMPTY);
	return true;
}

function ozsavelines($f,$lines,$error)
{
	$lines = implode("\r\n",$lines);
	return ozsavefile($f,$lines,$error);
}

function ozfileclear($f)
{
    if (!file_exists($f)) return;

	if (!isInValidDataDir($f))
	{
		$error = "File clear is not safe outside the data dir.";
		return false;
	}

	unlink($f);
}


function ozfilecountlist($file)
{
    if (!ozreadlines($file,$records,$error)) 
	{
        return 0;
    }

	return count($records);
}

function ozfileaddtolist($file,$rec, &$error)
{
	if (isempty($rec))
	{
		$error = "Cannot add an empty record to the list";
		return false;
	}

	if (!isInValidDataDir($file))
	{
		$error = "File add to list is not safe outside the data dir.";
		return false;
	}

    if (!ozreadlines($file,$records,$error)) 
	{
		$error = "File add to list failed. Cannot read lines. $error";
        return false;
    }

	if (in_array($rec,$records))
	{
		$error = "Record is already in the list: $rec";
		return true;
	}

	$records[] = $rec;
	$recordlist = implode("\r\n",$records)."\r\n";
	if (!file_put_contents($file,$recordlist))
	{
		$error = error_get_last()["message"];
		$error = "File add to list failed. Cannot save lines. $error";
		return false;
	}
	
	$error = "";
	return true;
}

function ozfiledelfromlist($file,$rec,&$error,$deleteifempty=true)
{
	if (!isInValidDataDir($file))
	{
		$error = "File del from list is not safe outside the data dir.";
		return false;
	}

	if (!ozreadlines($file,$records,$error)) 
	{
		return false;
	}

	if (!in_array($rec,$records))
	{
		return true;
	}
	
	$key = array_search($rec, $records);
	unset($records[$key]);
	$recordlist = implode("\r\n",$records);
	
	if ($deleteifempty&&count($records)==0)
	{
		if (!unlink($file)) { 
			$error = "Cannot delete file: $file";
			return false;
		} 
		return true;
	}

	if (file_put_contents($file,$recordlist)===false)
	{
		$error = "Could not write file: $file";
		return false;
	}

	return true;
}

function ozfileisinlist($file,$rec)
{
	if (!ozreadlines($file,$records,$error))
	{
		return false;
	}

	if (in_array($rec,$records))
	{
		return true;
	}
	return false;
}

function ozfiletypetoextension($filetype)
{
	switch($filetype)
	{
		case "image/jpeg" : return ".jpg";
		case "image/png" : return ".png";
		case "image/webp" : return ".webp";
		case "image/bmp" : return ".bmp";
		case "application/msword" : return ".doc";
		case "application/vnd.ms-excel": return ".xls";
		case "application/vnd.ms-powerpoint": return ".ppt";
		case "text/plain": return ".txt";
		case "application/pdf": return ".pdf";
		case "application/vnd.openxmlformats-officedocument.wordprocessingml.document": return ".docx";
		case "application/vnd.openxmlformats-officedocument.presentationml.slideshow": return ".pptx";
		case "application/vnd.openxmlformats-officedocument.presentationml.presentation": return ".pptx";
		case "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": return ".xlsx";
		case ".csv": return ".csv";
		default:return"";
	}
}

function ozfileextension($filename)
{
	$pos=strrpos($filename,".");
	if ($pos===false) return"";
	return substr($filename,$pos);
}

function ozfilesize($file,$unit="") 
{
	if (!file_exists($file)) return "0 byte";
	$filesize = filesize($file);
	return ozkbsize($filesize);
}

function ozkbsize($size,$unit="") 
{
	if( (!$unit && $size >= 1<<30) || $unit == "GB")
	  return number_format($size/(1<<30),2)."GB";
	if( (!$unit && $size >= 1<<20) || $unit == "MB")
	  return number_format($size/(1<<20),2)."MB";
	if( (!$unit && $size >= 1<<10) || $unit == "KB")
	  return number_format($size/(1<<10),2)."KB";
	return number_format($size)." bytes";
  }

//**********************************************************
// File lock
//**********************************************************

function ozfileaddtolist_uselock($f,$record,&$error)
{
    if(isempty($record))
    {
        $error="Cannot add an empty record to the list.";
        return false;
    }
    if(!ozfileopenread($f,"c+",LOCK_EX,$content,$fh,$error))return false;
    $records=preg_split("/\r\n|\r|\n/",$content,-1,PREG_SPLIT_NO_EMPTY);
    if(in_array($record,$records))
    {
        fclose($fh);
        return true;
    }
    $records[]=$record;
    $recordsjoined=implode("\r\n",$records)."\r\n";
    if(!ozfilewriteclose($fh,$recordsjoined,$error))return false;
    return true;
}

function ozfiledelfromlist_uselock($f,$record,&$error,$deleteifempty=true)
{
    if(!ozfileopenread($f,"c+",LOCK_EX,$content,$fh,$error))return false;
    $records=preg_split("/\r\n|\r|\n/",$content,-1,PREG_SPLIT_NO_EMPTY);
    $key=array_search($record,$records);
    if($key===false)
    {
        fclose($fh);
        return true;
    }
    unset($records[$key]);
    $recordsjoined=implode("\r\n",$records)."\r\n";
    if(!ozfilewriteclose($fh,$recordsjoined,$error))return false;
    if($deleteifempty&&count($records)==0)ozfileclear($f);
    return true;
}

function ozreadlines_uselock($f,&$lines,&$error,$timeout=1000)
{
	if(!ozreadfile_uselock($f,$content,$error,$timeout))return false;
	$lines=preg_split("/\r\n|\r|\n/",$content,-1,PREG_SPLIT_NO_EMPTY);
	return true;
}

function ozreadjson_uselock($f,&$decoded,&$error,$timeout=1000)
{
	if(!ozreadfile_uselock($f,$encoded,$error,$timeout))return false;
	$decoded=json_decode($encoded,true);
	if($decoded===null)
	{
		$error="JSON cannot be decoded.";
		return false;
	}
	return true;
}

function ozreadfile_uselock($f,&$content,&$error,$timeout=1000)
{
	if(!ozfileopenread($f,"r",LOCK_SH,$content,$fh,$error,$timeout))return false;
	fclose($fh);
	return true;
}

function ozfilelockread($f,$timeout=1000)
{
	if(!ozfileopenread($f,"r",LOCK_SH,$content,$fh,$error,$timeout))return false;
	fclose($fh);
	return$content;
}

function ozfilelock($f,&$error,$timeout=1000,$fopenmode="r")
{
	global$filelockregistry;
	if(!is_array($filelockregistry))$filelockregistry=[];
	if(array_key_exists($f,$filelockregistry))
	{
		$filelockregistry[$f]["lockcount"]+=1;
		return true;
	}
	$fh=@fopen($f,$fopenmode);
	if($fh===false)
	{
		$error="Could not open file for reading.";
		$errordetails=error_get_last();
		if(is_array($errordetails))$error.=" ".$errordetails["message"];
		return false;
	}
	$lockobtained=false;
	$timeelapsed=0;
	while(true)
	{
		$lockobtained=flock($fh,LOCK_EX|LOCK_NB);
		if($lockobtained)break;
		$timeelapsed+=200;
		if($timeelapsed>$timeout)break;
		usleep(200*1000);
	}
	if(!$lockobtained)
	{
		fclose($fh);
		$error="Could not obtain lock.";
		return false;
	}
	$entry["file"]=$f;
	$entry["filehandle"]=$fh;
	$entry["lockcount"]=1;
	$filelockregistry[$f]=$entry;
	return true;
}

function ozfileunlock($f)
{
	global$filelockregistry;
	if(!is_array($filelockregistry))$filelockregistry=[];
	if(!array_key_exists($f,$filelockregistry))return false;
	if(--$filelockregistry[$f]["lockcount"]>0)return true;
	fclose($filelockregistry[$f]["filehandle"]);
	unset($filelockregistry[$f]);
	return true;
}

function ozfileopenread($f,$fopenmode,$flockop,&$content,&$fh,&$error,$timeout=1000)
{
    if(!isInValidDataDir($f))
    {
        $error="File read is not safe outside the data dir.";
        return false;
    }
    $fh=@fopen($f,$fopenmode);
    if($fh===false)
    {
        $error="Cannot open file. ".error_get_last()["message"];
        return false;
    }
    $lockobtained=false;
    $timeelapsed=0;
    while(true)
    {
        $lockobtained=flock($fh,$flockop|LOCK_NB);
        if($lockobtained)break;
        $timeelapsed+=200;
        if($timeelapsed>$timeout)break;
        usleep(200*1000);
    }
    if(!$lockobtained)
    {
        fclose($fh);
        $error="Cannot obtain lock.";
        return false;
    }
    $filesize=@filesize($f);
    if($filesize===false)
    {
        fclose($fh);
        $error="Cannot get file size. ".error_get_last()["message"];
        return false;
    }
    if($filesize==0)
    {
        $content="";
        return true;
    }
    $content=@fread($fh,$filesize);
    if($content===false)
    {
        fclose($fh);
        $error="Cannot read file. ".error_get_last()["message"];
        return false;
    }
    return true;
}

function ozfilewriteclose($fh,$content,&$error)
{
    if(!ftruncate($fh,0))
    {
        fclose($fh);
        $error="Cannot truncate file length to 0.";
        return false;
    }
    if(!rewind($fh))
    {
        fclose($fh);
        $error="Cannot rewind file pointer position.";
        return false;
    }
    $bytes=@fwrite($fh,$content);
    if($bytes===false)
    {
        fclose($fh);
        $error="Cannot write file. ".error_get_last()["message"];
        return false;
    }
    fclose($fh);
    $len=strlen($content);
    if($bytes!=$len)
    {
        $error="Only $bytes of $len written, possibly out of free disk space.";
        return false;
    }
    return true;
}
?>