<?php
if (isapi()) return;

//******************************************************
// JSON request processing
//******************************************************

function getRequestHeaders() {
    $headers = array();
    foreach($_SERVER as $key => $value) {
        if (substr($key, 0, 5) <> 'HTTP_') {
            continue;
        }
        $header = str_replace(' ', '-', ucwords(str_replace('_', ' ', strtolower(substr($key, 5)))));
        $headers[$header] = $value;
    }
    return $headers;
}

function ozsignaturevalid($user,$postdata,$signature)
{
	if (strlen($signature)>0) 
		return true;
	return false;
}

function checkSignature($postdata,&$userid,&$station,&$error)
{
    global $regdir;
    $headers = getRequestHeaders();
    if (isset($headers["Ozeki-User"])) 
    {
        $userid = $headers["Ozeki-User"];
        $station = $headers["Ozeki-Station"];
        $signature = $headers["Ozeki-Signature"];
    }

    if (isempty($userid) || !ozfilenamesafe($userid,$useridsafe,$error)) 
    {
        $error = "Cannot find Ozeki-User HTTP header.";
        return false;
    }

    if (!isset($signature) || isempty($signature))
    {
        $error = "Cannot find Ozeki-Signature HTTP header.";
        return false;
    }

    $userkeyfile = "$regdir/$useridsafe/publickey.txt";
    if (!ozreadfile($userkeyfile,$publicKeyPem,$error))
    {
        $error = "Cannot read public key of $userid. $error";
        return false;
    }

    if (!importRsaPublicKeyFromPEM($publicKeyPem, $publicKey,$error))
    {
        $error = "Cannot import public key of $userid. $error";
        return false;
    }

    //echo "\r\n";
    //echo "Text to check: '$postdata'\r\n";
    //echo "Signature: '$signature'\r\n";
    if (!isSignatureValid($postdata,$signature,$publicKey,$error))
    {
        $error = "User authentication failed. $error";
        return false;
    }

    return true;
}

function parseJsonApiPost(&$action,&$req,&$error)
{
    $postdata = file_get_contents('php://input');
    $req = json_decode($postdata, true);
    if ($req==null || !is_array($req) || !array_key_exists("action",$req) || isEmpty($req["action"])) 
    {
        $error = "The request does not contain an action";
        return false;
    }
    return true;
}

function isValidTimestamp($req,&$error)
{
    if (isEmpty($req["timestamp"])) 
    {
        $error = "The request does not contain a timestamp";
        return false;
    }

    date_default_timezone_set('UTC');
    $requestDate = new DateTime($req["timestamp"]);
    $now = new DateTime();
    $days = ($requestDate->diff($now)->days);
    if ($days>1) //időzónák miatt az egy napnál régebbi requesteket dobjuk csak el
    {
        $nowstr = $now->format('Y-m-d H:i:s');
        $reqstr = $requestDate->format('Y-m-d H:i:s');
        $error = "The request timestamp suggests this request is outdated. Request: $reqstr. Now: $nowstr";
        return false;
    }
    return true;
}


function isValidSignedInput(&$userid,&$station,&$req,&$action,&$error)
{
    if (isset($_POST["json"]))
    {
        //Content type: application/x-www-form-urlencoded
        //Content type: multipart/form-data; boundary=...
        $postdata = $_POST["json"];
    } 
    else 
    {
        //Content type: application/json
        $postdata = file_get_contents('php://input');
    }

    if (!checkSignature($postdata,$userid,$station,$error)) return false;
    $req = json_decode($postdata, true);
    if (isEmpty($req["action"])) 
    {
        $error = "The request does not contain an action";
        return false;
    }

    //Valid timestamp
    if (!isValidTimestamp($req,$error))
    {
        return false;
    }

    $action = $req["action"];

    return true;
}

function isValidInput(&$userid,&$station,&$req,&$action,&$error)
{
    $headers = getRequestHeaders();
    $userid = $headers["Ozeki-User"];
    $station = $headers["Ozeki-Station"];

    $postdata = file_get_contents('php://input');

    $req = json_decode($postdata, true);
    if (isEmpty($req["action"])) 
    {
        $error = "The request does not contain an action";
        return false;
    }
    
    $action = $req["action"];

    return true;
}


function isValidInputWithStation(&$station,&$req,&$action,&$error)
{
    $headers = getRequestHeaders();
    $station = isset($headers["Ozeki-Station"]) ? $headers["Ozeki-Station"] : "";

    $postdata = file_get_contents('php://input');

    $req = json_decode($postdata, true);
    if (isEmpty($req["action"])) 
    {
        $error = "The request does not contain an action";
        return false;
    }

    if (isEmpty($station)) 
    {
        $error = "The request does not contain a station id";
        return true;
    }
    
    $action = $req["action"];

    $error = "";
    return true;
}

//******************************************************
// Client version code
//******************************************************

function getclientversioncode($req)
{
    if(!isset($req["clientversioncode"]))return 1;
    return intval($req["clientversioncode"]);
}

//******************************************************
// Request params
//******************************************************

function getreqparamboolopt($req,$paramname,&$paramval,&$resp,&$error,$defaultval)
{
    if(isset($req[$paramname]))
        return getreqparambool($req,$paramname,$paramval,$resp,$error);
    $paramval=$defaultval;
    return true;
}

function getreqparamstropt($req,$paramname,&$paramval,&$resp,&$error,$defaultval)
{
    if(isset($req[$paramname]))
        return getreqparamstr($req,$paramname,$paramval,$resp,$error);
    $paramval=$defaultval;
    return true;
}

function getreqparamstrarropt($req,$paramname,&$paramval,&$resp,&$error,$defaultval){
    if(isset($req[$paramname]))
        return getreqparamstrarr($req,$paramname,$paramval,$resp,$error);
    $paramval = $defaultval;
    return true;
}

function getreqparamstr($req,$paramname,&$paramval,&$resp,&$error)
{
    if(!isset($req[$paramname]))
        return ozerror(__FILE__,"Parameter is missing: $paramname",$resp,$error);
    if(!is_string($req[$paramname]))
        return ozerror(__FILE__,"Parameter is not a string: $paramname",$resp,$error);
    $paramval=$req[$paramname];
    return true;
}

function getreqparamint($req,$paramname,&$paramval,&$resp,&$error)
{
    if(!isset($req[$paramname]))
        return ozerror(__FILE__,"Parameter is missing: $paramname",$resp,$error);
    if(!is_integer($req[$paramname]))
        return ozerror(__FILE__,"Parameter is not an integer: $paramname",$resp,$error);
    $paramval=$req[$paramname];
    return true;
}

function getreqparambool($req,$paramname,&$paramval,&$resp,&$error)
{
    if(!isset($req[$paramname]))
        return ozerror(__FILE__,"Parameter is missing: $paramname",$resp,$error);
    $paramval=filter_var($req[$paramname],FILTER_VALIDATE_BOOLEAN,FILTER_NULL_ON_FAILURE);
    if($paramval===null)
        return ozerror(__FILE__,"Parameter is not a boolean: $paramname",$resp,$error);
    return true;
}

function getreqparamstrarr($req,$paramname,&$paramval,&$resp,&$error)
{
    if(!isset($req[$paramname]))
        return ozerror(__FILE__,"Parameter is missing: $paramname",$resp,$error);
    if(!is_array($req[$paramname]))
        return ozerror(__FILE__,"Parameter is not an array: $paramname",$resp,$error);
    foreach($req[$paramname] as $arrval)
    {
        if(!is_string($arrval))
            return ozerror(__FILE__,"Parameter is not a string array: $paramname",$resp,$error);
    }
    $paramval=$req[$paramname];
    return true;
}

function getreqparamtimestamp($req,$paramname,&$paramval,&$resp,&$error)
{
    if(!getreqparamstr($req,$paramname,$paramval,$resp,$error))return false;
    $paramval=strtotime($paramval);
    if($paramval===false)
        return ozerror(__FILE__,"Parameter is not a MySQL date string: $paramname",$resp,$error);
    return true;
}

//******************************************************
// Api error
//******************************************************

//Example:  returnError(__FILE__,"Invalid username or password.");
function returnError($source,$errormessage)
{
    ozerror($source, $errormessage,$resp,$error);
    $resp = json_encode($resp, JSON_PRETTY_PRINT);
    echo $resp;
    exit;
}


function ozerror($source, $errorMessage,&$resp,&$error,$errorExtra="")
{
    errorLog($source, $errorMessage, $errorExtra);
    $error = $errorMessage;
    $resp=[];
    $resp["status"] = "ERROR";
    $resp["errormessage"] = $errorMessage;
    return false;
}

function errorLog($caller,$error,$details="")
{
	global $lasterror;
    if (strlen($details)>0) $details = "; ".$details;
    ozlog($caller, "ERRO", $error.$details);
    if (isDebug()) error_log("api.php: $caller: $error");
	$lasterror = $error;
}

function getServiceNameFromFile($source)
{
    global $dirprefix;
    $source = str_replace("\\","/",$source);
    $dirpref = str_replace("\\","/",$dirprefix);
    $source = str_replace($dirpref,"",$source);
    $pathparts = explode("/",$source);
    $pathparts = explode("/",$source);
    $pcnt = count($pathparts);
    if ($pcnt>1)
        return $pathparts[2];
    return "";
}

function ozlogdir($source)
{
    global $datadir,$pageservice;
    $srv = getServiceNameFromFile($source);
    if ($srv!="")
    {
        $logdir = ozpathcombine($datadir,"logs",$srv);
    } else {
        $logdir = ozpathcombine($datadir,"logs",$pageservice[$owpn]);
    }
    return $logdir;
}

function ozlog($source,$level,$message,$logfile=null)
{
	global $datadir,$pageservice,$owpn;

    if (function_exists("ozlogscreen"))
    {
        ozlogscreen($source,$level,$message,$logfile);
    }

    if ($logfile == null) 
    {
        $logdir = ozlogdir($source);
        if (!ozmkdir($logdir,$error)) return;
        $logfile = "$logdir/log.txt";
        ozlogrotation($source,$logfile);
    } else 
    {
        $logdir = dirname($logfile);
        if (!ozmkdir($logdir,$error)) return;
    }
    $src = basename($source);
    $source = str_replace("\\","/",$source);
    $srcpath = substr($source,0,strlen($source)-strlen($src)-1);
    $lastdir = substr($srcpath,strrpos($srcpath,"/"));
    $message = trim($message);
    $ip = (isset($_SERVER['REMOTE_ADDR'])) ? " ".$_SERVER['REMOTE_ADDR'] : " 0.0.0.0";
	$entry = date('Y-m-d H:i:s').$ip." $lastdir/$src $level $message\r\n";
    $logfileexists=is_file($logfile);
	file_put_contents($logfile, $entry, FILE_APPEND | LOCK_EX);
    if(!$logfileexists && isLinux())
    {
        $cmd="/bin/chown www-data.www-data $logfile";
        passthru($cmd);
    }
}

function ozlogrotation($source,$logfile)
{
    global$pageservice,$owpn;
    $srv=getServiceNameFromFile($source);
    if($srv=="")$srv=$pageservice[$owpn];
    $maxsize=ozlogrotation_getmaxsize($srv);
    if(!@is_file($logfile))return;
    $size=@filesize($logfile);
    if(!$size)return;
    if($size<$maxsize)return;
    $logdir=dirname($logfile);
    if(!ozfilelock("$logdir/logrotation.lock",$error,1000,"c"))return;
    try
    {
        if(!@is_file($logfile))return;
        $size=@filesize($logfile);
        if(!$size)return;
        if($size<$maxsize)return;
        $maxcount=ozlogrotation_getmaxcount($srv);
        ozrotatelogfiles($logdir,$maxcount);
    }
    finally
    {
        ozfileunlock("$logdir/logrotation.lock");
    }
}

function ozlogrotation_getmaxsize($servicename)
{
    $maxsize=5000000;
    if(isset($GLOBALS["logrotation_size"]))
        $maxsize=$GLOBALS["logrotation_size"];
    if(isset($GLOBALS["logrotation_".$servicename."_size"]))
        $maxsize=$GLOBALS["logrotation_".$servicename."_size"];
    if(!is_integer($maxsize))
        $maxsize=5000000;
    if($maxsize<1000000)
        $maxsize=1000000;
    return$maxsize;
}

function ozlogrotation_getmaxcount($servicename)
{
    $maxcount=5;
    if(isset($GLOBALS["logrotation_count"]))
        $maxcount=$GLOBALS["logrotation_count"];
    if(isset($GLOBALS["logrotation_".$servicename."_count"]))
        $maxcount=$GLOBALS["logrotation_".$servicename."_count"];
    if(!is_integer($maxcount))
        $maxcount=5;
    if($maxcount<1)
        $maxcount=1;
    return$maxcount;
}

function ozrotatelogfiles($logdir,$maxcount)
{
    if(!ozlsdir($logdir,$files,$dirs,$error))return;
    $detailsarr=ozgetlogfiledetails($files);
    usort($detailsarr,function($a,$b){
        return$a["order"]-$b["order"];
    });
    for($i=count($detailsarr)-1;$i>=0;$i--)
    {
        $d=$detailsarr[$i];
        $oldname="$logdir/".$d["filename"];
        if($i>$maxcount-1)
        {
            if(!@unlink($oldname))break;
            continue;
        }
        $neworder=$d["order"]+1;
        $newname="$logdir/log.$neworder.txt";
        if(!@rename($oldname,$newname))break;
    }
}

function ozgetlogfiledetails($files)
{
    $detailsarr=[];
    foreach($files as $f)
    {
        if(preg_match("/^log(?:\.([1-9][0-9]*))?\.txt$/",$f,$matches,PREG_UNMATCHED_AS_NULL)!==1)continue;
        $d=[];
        $d["filename"]=$f;
        $d["order"]=intval($matches[1]);
        $detailsarr[]=$d;
    }
    return$detailsarr;
}

//******************************************************
// API client
//******************************************************

function createHeaders($userhash,$password,$postdata,&$headers,&$error)
{
    global $regdir, $chatclient_stationid;
    
    $publickeyfile = "$regdir/$userhash/privatekey_".sha512($password).".txt";
    
    if (!ozreadfile($publickeyfile,$privateKeyPem,$error) || isempty($privateKeyPem)) 
    {
        $error = "Could not read public key of $userhash. $error";
        return false;
    }
    
    if (!importRsaPrivateKeyFromPEM($privateKeyPem, $password, $privateKey, $publicKey,$error))
    {
        $error = "Could not import public key of $userhash. $error";
        return false;
    }

    if (!signRSA($privateKey,$postdata,$signature,$error))
    {
        $error = "Could not sign post data. $error";
        return false;
    }

    $headers = [];
    $headers["Ozeki-User"] = $userhash;
    $headers["Ozeki-Signature"] = $signature;
    $headers["Ozeki-Station"] = $chatclient_stationid;

    return true;
}

function postJSON($server,$json,&$response,&$error,$headers=[])
{
    if (isEmpty($server))
    {
        $error = "Server address not specified.";
        return false;
    }

    $headers["Content-Type"] = "application/json";
    $headers["Content-Length"] = strlen($json);
    $header = "";
    foreach($headers as $key=>$value)
    {
        $header .= $key.": ".$value."\r\n";
    }

    $context_options['http']['method'] = 'POST';
    $context_options['http']['header'] = $header;
    $context_options['http']['content'] = $json;
    $context_options['http']['ignore_errors'] = true;

    $context = stream_context_create($context_options); 

    if (($response = @file_get_contents($server, false, $context)) === false) 
	{
		$error = error_get_last();
        if (is_array($error)) $error = $error["message"];
	    return false;
  	}
    return true;
}

function postRequest($server,$data,&$response,&$error,$headers=[])
{
    $header = "";
    foreach($headers as $key=>$value)
    {
        $header .= $key.": ".$value."\r\n";
    }

    $context_options['http']['method'] = 'POST';
    $context_options['http']['header'] = $header;
    $context_options['http']['content'] = $data;
    $context_options['http']['ignore_errors'] = true;

    $context = stream_context_create($context_options); 

    if (($response = @file_get_contents($server, false, $context)) === false) 
	{
		$error = error_get_last();
	    return false;
  	}
    return true;
}

function getRequest($url,&$response,&$error,$headers=[])
{
    $header = "";
    foreach($headers as $key=>$value)
    {
        $header .= $key.": ".$value."\r\n";
    }

    $context_options['http']['method'] = 'GET';
    $context_options['http']['header'] = $header;
    $context_options['http']['ignore_errors'] = true;

    $context = stream_context_create($context_options); 

    if (($response = @file_get_contents($url, false, $context)) === false) 
	{
		$error = error_get_last();
	    return false;
  	}
    return true;
}

//******************************************************
// API testing
//******************************************************
function displayHeaders($headers)
{
    $ret = "";
    if (!is_array($headers)) return "";
    foreach($headers as $key=>$val)
    {
        $ret .= "<b>$key:</b> ".ozfrontwithdots($val,100)."<BR>";
    }
    return $ret;
}

function displayRequestAndResponse($request,$requestheaders,$response)
{
    $resp = "<HR>";
    $resp .= "<h2>Request</h2>";
    $resp .= displayHeaders($requestheaders);
    $resp .= "<textarea style='width:800px; height:100px;'>";
    $resp .= $request;
    $resp .= "</textarea>";
    //$resp .= jsonToDebug($request);
    
    $resp .= "<HR>";
    $resp .= "<h2>Response</h2>";
    $resp .= "<textarea style='width:800px; height:200px;'>";
    $resp .= $response;
    $resp .= "</textarea>";
    //$resp .= jsonToDebug($response);

    return $resp;
}

//******************************************************
// Allow cross site scripting
//******************************************************

function ozCors() {
    
    // Allow from any origin
    if (isset($_SERVER['HTTP_ORIGIN'])) {
        // Decide if the origin in $_SERVER['HTTP_ORIGIN'] is one
        // you want to allow, and if so:
        header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
		//header("Access-Control-Allow-Origin: *");
        header('Access-Control-Allow-Credentials: true');
        header('Access-Control-Max-Age: 86400');    // cache for 1 day
    }
    
    // Access-Control headers are received during OPTIONS requests
    if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
        
        if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD']))
            // may also be using PUT, PATCH, HEAD etc
            header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
        
        if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']))
            header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");
    
        exit(0);
    }
}

//******************************************************
// Load api service
//******************************************************

if ($api!="" && $srv!= "")
{  
    if (!ozfilenamesafe($srv,$safesrvname,$error)) 
    {
        echo "API ERROR. Unsafe srv request."; 
        exit;
    }
    
    if (!file_exists("$dirprefix/ozekiservices/$safesrvname/_api.php"))
    {
        echo "API ERROR. _api.php not found for service $srv."; 
        exit;
    }    

    $srv = $safesrvname;
    include_once("$dirprefix/ozekiservices/$safesrvname/_api.php");
}

//******************************************************
// Load system service
//******************************************************
if ($owpn!="" && $srv!="" && 
    ozfilenamesafe($srv,$safesrvname,$error) &&
    file_exists("$dirprefix/ozekiservices/$safesrvname/_include.php"))    
{
    include_once("$dirprefix/ozekiservices/$safesrvname/_include.php");
}

ozCors();

?>